const bcrypt = require("bcryptjs")
const jwt = require("jsonwebtoken")
const crypto = require("crypto")

class AuthController {
  constructor() {
    this.signup = this.signup.bind(this)
    this.login = this.login.bind(this)
    this.getProfile = this.getProfile.bind(this)
    this.updateProfile = this.updateProfile.bind(this)
    this.logout = this.logout.bind(this)
  }

  // Generate organization ID using the user's database ID
  generateOrgId(userId) {
    return `org${userId}`
  }

  async signup(req, res) {
    const { firstname, lastname, email, username, password } = req.body

    try {
      // Validation
      if (!firstname || !lastname || !email || !username || !password) {
        return res.status(400).json({
          success: false,
          message: "All fields are required",
        })
      }

      // Email validation
      const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
      if (!emailRegex.test(email)) {
        return res.status(400).json({
          success: false,
          message: "Invalid email format",
        })
      }

      // Password validation
      if (password.length < 6) {
        return res.status(400).json({
          success: false,
          message: "Password must be at least 6 characters long",
        })
      }

      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Check if user already exists
      const [existingUsers] = await empCloudDbPool.execute(
        "SELECT id FROM useraccounts WHERE email = ? OR username = ?",
        [email, username],
      )

      if (existingUsers.length > 0) {
        return res.status(400).json({
          success: false,
          message: "User with this email or username already exists",
        })
      }

      // Hash password
      const saltRounds = 12
      const hashedPassword = await bcrypt.hash(password, saltRounds)

      // Create user first without orgid (will be updated after we get the user ID)
      const [result] = await empCloudDbPool.execute(
        `INSERT INTO useraccounts (firstname, lastname, email, username, password, datecreated) 
         VALUES (?, ?, ?, ?, ?, NOW())`,
        [firstname, lastname, email, username, hashedPassword],
      )

      const userId = result.insertId

      // Generate organization ID using the actual user ID
      const orgId = this.generateOrgId(userId)

      // Update the user record with the generated orgid
      await empCloudDbPool.execute("UPDATE useraccounts SET orgid = ? WHERE id = ?", [orgId, userId])

      console.log(`✅ User created successfully with ID: ${userId}, OrgID: ${orgId}`)

      // Create product record with default services (flag1=1, flag2=1)
      try {
        const productName = `${firstname} ${lastname} Organization` // You can customize this
        await empCloudDbPool.execute(
          `INSERT INTO product (loginid, productname, flag1, flag2, flag3, flag4, flag5, flag6, flag7, flag8, datecreated) 
           VALUES (?, ?, 1, 1, 0, 0, 0, 0, 0, 0, NOW())`,
          [userId, productName],
        )

        console.log(`✅ Product record created for user ID: ${userId} with default services (flag1=1, flag2=1)`)
      } catch (productError) {
        console.error(`⚠️ Failed to create product record for user ID: ${userId}:`, productError.message)
        // Don't fail the signup process if product creation fails
      }

      // Hit .NET API after successful user creation - Using the actual user ID
      let dotNetApiResult = {
        status: "PENDING",
        message: "API call not attempted",
      }

      try {
        console.log(`🔄 Calling .NET API for user ID: ${userId}`)
        console.log(`🌐 API URL: http://103.148.165.134:5006/api/Home/init?param=${userId}`)

        const fetch = require("node-fetch")
        const dotNetResponse = await fetch(`http://103.148.165.134:5006/api/Home/init?param=${userId}`, {
          method: "GET",
          headers: {
            "Content-Type": "application/json",
            "User-Agent": "Employee-Monitoring-System/1.0",
          },
          timeout: 15000, // 15 second timeout
        })

        const responseText = await dotNetResponse.text()
        console.log(`📊 .NET API Response Status: ${dotNetResponse.status}`)
        console.log(`📄 .NET API Response Body: ${responseText}`)

        if (dotNetResponse.ok) {
          console.log("✅ .NET API call successful - System initialized")

          dotNetApiResult = {
            status: "OK",
            message: "System initialized successfully",
            response: responseText,
            apiUrl: `http://103.148.165.134:5006/api/Home/init?param=${userId}`,
            userId: userId,
          }
        } else {
          console.error(`❌ .NET API call failed with status: ${dotNetResponse.status}`)
          console.error(`📄 Error Response: ${responseText}`)

          dotNetApiResult = {
            status: "ERROR",
            message: `API returned status ${dotNetResponse.status}`,
            error: responseText,
            apiUrl: `http://103.148.165.134:5006/api/Home/init?param=${userId}`,
            userId: userId,
          }
        }
      } catch (dotNetError) {
        console.error("❌ .NET API call error:", dotNetError.message)
        console.error("🔍 Error details:", dotNetError)

        dotNetApiResult = {
          status: "ERROR",
          message: "Network error calling .NET API",
          error: dotNetError.message,
          apiUrl: `http://103.148.165.134:5006/api/Home/init?param=${userId}`,
          userId: userId,
        }
      }

      // Always return success for user creation, include .NET API status
      res.status(201).json({
        success: true,
        message: "User created successfully",
        data: {
          id: userId,
          firstname,
          lastname,
          email,
          username,
          orgId,
          dotNetApi: dotNetApiResult,
        },
      })
    } catch (error) {
      console.error("❌ Signup error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  async login(req, res) {
    const { username, password } = req.body

    try {
      // Validation
      if (!username || !password) {
        return res.status(400).json({
          success: false,
          message: "Username and password are required",
        })
      }

      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Find user by username or email
      const [users] = await empCloudDbPool.execute("SELECT * FROM useraccounts WHERE username = ? OR email = ?", [
        username,
        username,
      ])

      if (users.length === 0) {
        return res.status(401).json({
          success: false,
          message: "Invalid credentials",
        })
      }

      const user = users[0]

      // Verify password
      const isPasswordValid = await bcrypt.compare(password, user.password)
      if (!isPasswordValid) {
        return res.status(401).json({
          success: false,
          message: "Invalid credentials",
        })
      }

      // Generate JWT token
      const token = jwt.sign(
        {
          userId: user.id,
          username: user.username,
          orgId: user.orgid,
        },
        process.env.JWT_SECRET || "your-secret-key",
        { expiresIn: "24h" },
      )

      // Remove password from user object
      const { password: _, ...userWithoutPassword } = user

      console.log(`✅ User logged in successfully: ${user.username} (ID: ${user.id})`)

      res.json({
        success: true,
        message: "Login successful",
        token,
        user: userWithoutPassword,
      })
    } catch (error) {
      console.error("❌ Login error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  async getProfile(req, res) {
    try {
      const empCloudDbPool = req.app.locals.empCloudDbPool

      const [users] = await empCloudDbPool.execute(
        "SELECT id, firstname, lastname, email, username, orgid, link, datecreated FROM useraccounts WHERE id = ?",
        [req.user.userId],
      )

      if (users.length === 0) {
        return res.status(404).json({
          success: false,
          message: "User not found",
        })
      }

      res.json({
        success: true,
        user: users[0],
      })
    } catch (error) {
      console.error("Get profile error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  async updateProfile(req, res) {
    const { firstname, lastname, email } = req.body

    try {
      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Check if email is already taken by another user
      if (email) {
        const [existingUsers] = await empCloudDbPool.execute(
          "SELECT id FROM useraccounts WHERE email = ? AND id != ?",
          [email, req.user.userId],
        )

        if (existingUsers.length > 0) {
          return res.status(400).json({
            success: false,
            message: "Email is already taken",
          })
        }
      }

      // Update user profile
      await empCloudDbPool.execute("UPDATE useraccounts SET firstname = ?, lastname = ?, email = ? WHERE id = ?", [
        firstname,
        lastname,
        email,
        req.user.userId,
      ])

      res.json({
        success: true,
        message: "Profile updated successfully",
      })
    } catch (error) {
      console.error("Update profile error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  // Method to update the .exe file link (can be called separately when needed)
  async updateExeLink(req, res) {
    const { exeLink } = req.body

    try {
      if (!exeLink) {
        return res.status(400).json({
          success: false,
          message: "Exe link is required",
        })
      }

      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Update the link field with the .exe file link
      await empCloudDbPool.execute("UPDATE useraccounts SET link = ? WHERE id = ?", [exeLink, req.user.userId])

      console.log(`✅ Exe link updated for user ID: ${req.user.userId}`)

      res.json({
        success: true,
        message: "Exe link updated successfully",
        data: {
          userId: req.user.userId,
          exeLink: exeLink,
        },
      })
    } catch (error) {
      console.error("❌ Update exe link error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  async logout(req, res) {
    res.json({
      success: true,
      message: "Logged out successfully",
    })
  }
}

module.exports = new AuthController()
