const errorHandler = (err, req, res, next) => {
  console.error('Error Stack:', err.stack);
  
  // Database connection errors
  if (err.code === 'ER_ACCESS_DENIED_ERROR') {
    return res.status(500).json({
      success: false,
      message: 'Database connection failed',
      error: 'Access denied to database'
    });
  }
  
  if (err.code === 'ER_BAD_DB_ERROR') {
    return res.status(500).json({
      success: false,
      message: 'Database not found',
      error: 'Invalid database name'
    });
  }
  
  if (err.code === 'ECONNREFUSED') {
    return res.status(500).json({
      success: false,
      message: 'Database connection refused',
      error: 'Cannot connect to database server'
    });
  }
  
  // MySQL specific errors
  if (err.code === 'ER_NO_SUCH_TABLE') {
    return res.status(500).json({
      success: false,
      message: 'Database table not found',
      error: 'Required table does not exist'
    });
  }
  
  if (err.code === 'ER_DUP_ENTRY') {
    return res.status(409).json({
      success: false,
      message: 'Duplicate entry',
      error: 'Record already exists'
    });
  }
  
  // Validation errors
  if (err.name === 'ValidationError') {
    return res.status(400).json({
      success: false,
      message: 'Validation failed',
      error: err.message
    });
  }
  
  // JWT errors (for future use)
  if (err.name === 'JsonWebTokenError') {
    return res.status(401).json({
      success: false,
      message: 'Invalid token',
      error: 'Authentication failed'
    });
  }
  
  if (err.name === 'TokenExpiredError') {
    return res.status(401).json({
      success: false,
      message: 'Token expired',
      error: 'Please login again'
    });
  }
  
  // Default error
  res.status(err.status || 500).json({
    success: false,
    message: err.message || 'Internal server error',
    error: process.env.NODE_ENV === 'production' 
      ? 'Something went wrong' 
      : err.message || 'Unknown error occurred'
  });
};

module.exports = errorHandler;