const express = require("express")
const router = express.Router()
const { getServices, updateServices, getServiceStatus } = require("../controllers/serviceController")
const authMiddleware = require("../middleware/authMiddleware.js")
const fetch = require("node-fetch")

// All routes require authentication
router.use(authMiddleware.verifyToken)

// Service management routes
router.get("/", getServices)
router.put("/", updateServices)
router.get("/status", getServiceStatus)

// Add this new route for fetching screenshots
router.post("/fetch-screenshots", async (req, res) => {
  try {
    const { loginid, userid, startDate, endDate, userDetailsId, serviceId } = req.body
    const adminId = req.user.userId

    console.log(`📡 Screenshot API Request:`, {
      loginid,
      userid,
      startDate,
      endDate,
      userDetailsId,
      serviceId,
      adminId,
    })

    // First check if we have data in the database table
    const adminDb = req.adminDb
    const tableName = `userdata${userDetailsId}${serviceId}`

    console.log(`🔍 Checking table: ${tableName}`)

    const [rows] = await adminDb.execute(
      `SELECT COUNT(*) as count FROM ${tableName} WHERE loginid = ? AND DATE(startdatetime) >= ? AND DATE(startdatetime) <= ?`,
      [loginid, startDate, endDate],
    )

    const recordCount = rows[0].count
    console.log(`📊 Found ${recordCount} records in database`)

    if (recordCount === 0) {
      return res.json({
        success: false,
        message: `No activity data found in ${tableName} for the specified date range`,
        type: "no_database_records",
      })
    }

    // Call the .NET API
    const netApiUrl = `http://your-dotnet-api-url/api/screenshots` // Replace with actual .NET API URL

    console.log(`📡 Calling .NET API: ${netApiUrl}`)

    const netApiResponse = await fetch(netApiUrl, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        loginid,
        userid,
        startDate,
        endDate,
      }),
    })

    console.log(`📡 .NET API Response Status: ${netApiResponse.status}`)

    if (netApiResponse.ok) {
      const contentType = netApiResponse.headers.get("content-type")

      if (contentType && contentType.includes("application/zip")) {
        // Forward the ZIP file
        console.log(`📦 Forwarding ZIP file from .NET API`)

        const buffer = await netApiResponse.arrayBuffer()

        res.setHeader("Content-Type", "application/zip")
        res.setHeader(
          "Content-Disposition",
          `attachment; filename="screenshots_${loginid}_${startDate}_${endDate}.zip"`,
        )
        res.send(Buffer.from(buffer))
      } else {
        // Forward JSON response
        const jsonData = await netApiResponse.json()
        console.log(`📋 .NET API JSON Response:`, jsonData)
        res.json(jsonData)
      }
    } else {
      const errorText = await netApiResponse.text()
      console.log(`❌ .NET API Error: ${errorText}`)

      res.status(netApiResponse.status).json({
        success: false,
        message: `NET API Error: ${errorText}`,
        type: "net_api_error",
        statusCode: netApiResponse.status,
      })
    }
  } catch (error) {
    console.error("❌ Screenshot fetch error:", error)
    res.status(500).json({
      success: false,
      message: "Failed to fetch screenshots",
      error: error.message,
      type: "server_error",
    })
  }
})

module.exports = router
