const fetch = require("node-fetch")

class ApiHelper {
  constructor() {
    this.baseUrl = process.env.DOTNET_API_BASE_URL || "http://103.148.165.134:5006"
    this.timeout = 180000 // Increased to 3 minutes timeout
  }

  async fetchWithTimeout(url, options = {}) {
    const controller = new AbortController()
    const timeoutId = setTimeout(() => controller.abort(), this.timeout)

    try {
      const response = await fetch(url, {
        ...options,
        signal: controller.signal,
        timeout: this.timeout, // Explicitly set node-fetch timeout
      })
      clearTimeout(timeoutId)
      return response
    } catch (error) {
      clearTimeout(timeoutId)
      if (error.name === "AbortError" || error.type === "body-timeout") {
        throw new Error(`Request timeout after ${this.timeout / 1000} seconds`)
      }
      throw error
    }
  }

  async fetchScreenshots(loginid, userid, date) {
    const url = `${this.baseUrl}/api/Home/GetImage?lid=${loginid}&uid=${userid}&date=${date}`

    console.log(`🌐 Calling external API: ${url}`)

    try {
      const response = await this.fetchWithTimeout(url, {
        method: "GET",
        headers: {
          Accept: "application/zip, application/octet-stream",
        },
        timeout: this.timeout, // Explicit timeout for node-fetch
      })

      console.log(`📡 API Response Status: ${response.status} ${response.statusText}`)

      if (!response.ok) {
        throw new Error(`API returned ${response.status}: ${response.statusText}`)
      }

      return response
    } catch (error) {
      console.error(`❌ API call failed: ${error.message}`)
      throw error
    }
  }

  async testConnection() {
    try {
      const response = await this.fetchWithTimeout(`${this.baseUrl}/api/health`, {
        method: "GET",
        timeout: 30000, // Shorter timeout for health check
      })
      return response.ok
    } catch (error) {
      console.error("API connection test failed:", error.message)
      return false
    }
  }
}

module.exports = new ApiHelper()
