const express = require("express")
const cors = require("cors")
const helmet = require("helmet")
// const rateLimit = require("express-rate-limit")
require("dotenv").config()
const path = require("path")
const fs = require("fs")
const databaseConfig = require("./config/database")

const app = express()
const PORT = process.env.PORT || 4000

// Enhanced error handling for uncaught exceptions
process.on("uncaughtException", (error) => {
  console.error("❌ Uncaught Exception:", error)
  console.error("Stack:", error.stack)
  // Don't exit immediately, log the error and continue
})

process.on("unhandledRejection", (reason, promise) => {
  console.error("❌ Unhandled Rejection at:", promise, "reason:", reason)
  // Don't exit immediately, log the error and continue
})

// Security middleware
app.use(
  helmet({
    crossOriginEmbedderPolicy: false,
    contentSecurityPolicy: false,
  }),
)

// if (process.env.ENABLE_RATE_LIMIT === "true") {
//   const limiter = rateLimit({
//     windowMs: 15 * 60 * 1000,
//     max: 200,
//     message: "Too many requests from this IP, please try again later.",
//     standardHeaders: true,
//     legacyHeaders: false,
//   });
//   app.use(limiter);
//   console.log("✅ Rate limiting enabled.");
// } else {
//   console.log("⚙️ Rate limiting disabled.");
// }


// CORS configuration with enhanced options
app.use(
  cors({
    origin: [
      process.env.FRONTEND_URL || "http://localhost:3000",
      "http://localhost:5173", // Vite default port
      "http://localhost:3000", // React default port
      "http://127.0.0.1:3000",
      "http://127.0.0.1:5173",
      "http://103.139.58.182",
    ],
    credentials: true,
    methods: ["GET", "POST", "PUT", "DELETE", "OPTIONS"],
    allowedHeaders: ["Content-Type", "Authorization", "X-Requested-With"],
  }),
)

// Request logging middleware for debugging
app.use((req, res, next) => {
  const timestamp = new Date().toISOString()
  console.log(`🔍 ${timestamp} - ${req.method} ${req.url}`)

  // Log request body for POST/PUT requests (excluding sensitive data)
  if ((req.method === "POST" || req.method === "PUT") && req.url !== "/api/auth/login") {
    console.log("📦 Request body keys:", Object.keys(req.body || {}))
  }

  next()
})

// ✅ BODY PARSING MIDDLEWARE - MUST COME BEFORE ROUTES
app.use(express.json({ limit: "10mb" }))
app.use(express.urlencoded({ extended: true, limit: "10mb" }))

// Create uploads directory if it doesn't exist
const uploadsDir = path.join(__dirname, "uploads", "csv")
if (!fs.existsSync(uploadsDir)) {
  fs.mkdirSync(uploadsDir, { recursive: true })
  console.log("📁 Created uploads/csv directory")
}

// Database connection with retry logic
let empCloudDbPool, edb3Pool

const initializeDatabases = async () => {
  try {
    console.log("🔄 Initializing database connections...")
    const pools = databaseConfig.createPools()
    empCloudDbPool = pools.empCloudDbPool
    edb3Pool = pools.edb3Pool

    // Make database pools available to routes
    app.locals.empCloudDbPool = empCloudDbPool
    app.locals.edb3Pool = edb3Pool

    // Test database connection
    await databaseConfig.testConnection()
    console.log("✅ Database connections established successfully")
  } catch (error) {
    console.error("❌ Database initialization failed:", error)
    console.log("🔄 Retrying database connection in 5 seconds...")

    setTimeout(initializeDatabases, 5000)
  }
}

// Initialize databases
initializeDatabases()

// Enhanced health check endpoint
app.get("/api/health", async (req, res) => {
  const startTime = Date.now()

  try {
    // Test database connectivity
    let dbStatus = "disconnected"
    let dbError = null

    if (empCloudDbPool) {
      try {
        await empCloudDbPool.execute("SELECT 1")
        dbStatus = "connected"
      } catch (error) {
        dbStatus = "error"
        dbError = error.message
      }
    }

    const responseTime = Date.now() - startTime
    const uptime = process.uptime()
    const memoryUsage = process.memoryUsage()

    res.json({
      success: true,
      message: "Server is running",
      status: {
        database: dbStatus,
        dbError: dbError,
        port: PORT,
        environment: process.env.NODE_ENV || "development",
        uptime: `${Math.floor(uptime / 60)}m ${Math.floor(uptime % 60)}s`,
        responseTime: `${responseTime}ms`,
        memory: {
          used: `${Math.round(memoryUsage.heapUsed / 1024 / 1024)}MB`,
          total: `${Math.round(memoryUsage.heapTotal / 1024 / 1024)}MB`,
        },
      },
      timestamp: new Date().toISOString(),
    })
  } catch (error) {
    console.error("❌ Health check error:", error)
    res.status(500).json({
      success: false,
      message: "Health check failed",
      error: error.message,
      timestamp: new Date().toISOString(),
    })
  }
})

// Routes with error handling
const routeHandler = (routePath, routeFile) => {
  try {
    return require(routeFile)
  } catch (error) {
    console.error(`❌ Error loading route ${routePath}:`, error.message)
    return (req, res) => {
      res.status(500).json({
        success: false,
        message: `Route ${routePath} is temporarily unavailable`,
        error: process.env.NODE_ENV === "development" ? error.message : "Internal server error",
      })
    }
  }
}

// Apply routes
app.use("/api/auth", routeHandler("/api/auth", "./routes/authRoutes"))
app.use("/api/employees", routeHandler("/api/employees", "./routes/employeeRoutes"))
app.use("/api/services", routeHandler("/api/services", "./routes/serviceRoutes"))
app.use("/api/test", routeHandler("/api/test", "./routes/testRoutes"))
app.use("/api/user-details", routeHandler("/api/user-details", "./routes/userDetailsRoutes"))
app.use("/api/user-activity", routeHandler("/api/user-activity", "./routes/userActivityRoutes"))

console.log("🔧 Registering department routes at /api/departments")
const departmentRoutes = routeHandler("/api/departments", "./routes/departmentRoutes")
app.use("/api/departments", departmentRoutes)

console.log("✅ Department routes registered successfully")

// ✅ ERROR HANDLING MIDDLEWARE - MUST COME AFTER ROUTES
app.use((err, req, res, next) => {
  if (err instanceof SyntaxError && err.status === 400 && "body" in err) {
    console.error("❌ Invalid JSON in request body:", err.message)
    return res.status(400).json({
      success: false,
      message: "Invalid JSON format",
      error: "Request body contains invalid JSON",
    })
  }
  next(err)
})

// Enhanced error handling middleware
app.use((err, req, res, next) => {
  const timestamp = new Date().toISOString()
  console.error(`❌ ${timestamp} - Error in ${req.method} ${req.url}:`, err.stack)

  // Database connection errors
  if (err.code === "PROTOCOL_CONNECTION_LOST" || err.code === "ECONNREFUSED") {
    console.log("🔄 Database connection lost, attempting to reconnect...")
    initializeDatabases()
  }

  res.status(err.status || 500).json({
    success: false,
    message: err.message || "Something went wrong!",
    error:
      process.env.NODE_ENV === "development"
        ? {
            stack: err.stack,
            code: err.code,
          }
        : {},
    timestamp: timestamp,
  })
})

// 404 handler
app.use("*", (req, res) => {
  console.log(`❌ 404 - Route not found: ${req.method} ${req.originalUrl}`)
  res.status(404).json({
    success: false,
    message: `Route not found: ${req.method} ${req.originalUrl}`,
    availableRoutes: [
      "GET /api/health",
      "POST /api/auth/login",
      "POST /api/auth/signup",
      "GET /api/user-details",
      "POST /api/user-details/screenshots",
      "GET /api/employees",
      "GET /api/services",
      "GET /api/departments",
      "POST /api/departments",
      "PUT /api/departments/:id",
      "DELETE /api/departments/:id",
    ],
  })
})

// Graceful shutdown handling
const gracefulShutdown = (signal) => {
  console.log(`\n🛑 Received ${signal}. Starting graceful shutdown...`)

  if (empCloudDbPool) {
    empCloudDbPool.end(() => {
      console.log("📊 EmpCloud database pool closed")
    })
  }

  if (edb3Pool) {
    edb3Pool.end(() => {
      console.log("📊 EDB3 database pool closed")
    })
  }

  process.exit(0)
}

process.on("SIGTERM", () => gracefulShutdown("SIGTERM"))
process.on("SIGINT", () => gracefulShutdown("SIGINT"))

// Start server with enhanced logging
const server = app.listen(PORT, () => {
  console.log(`🚀 Server is running on port ${PORT}`)
  console.log(`📊 Database: ${process.env.DB_NAME || "Not specified"}`)
  console.log(`🌍 Environment: ${process.env.NODE_ENV || "development"}`)
  console.log(`🔗 Health check: http://localhost:${PORT}/api/health`)
  console.log(`📅 Server started at: ${new Date().toISOString()}`)
})

// Handle server errors
server.on("error", (error) => {
  if (error.code === "EADDRINUSE") {
    console.error(`❌ Port ${PORT} is already in use`)
    process.exit(1)
  } else {
    console.error("❌ Server error:", error)
  }
})

module.exports = app
