// Get current user's service configuration
const getServices = async (req, res) => {
  try {
    const userId = req.user.userId
    const empCloudDbPool = req.app.locals.empCloudDbPool

    console.log(`🔍 Fetching services for user ID: ${userId}`)

    // Get user's product/service configuration
    const [services] = await empCloudDbPool.execute(
      `SELECT 
        id,
        loginid,
        productname,
        flag1,
        flag2,
        flag3,
        flag4,
        flag5,
        flag6,
        flag7,
        flag8,
        datecreated
      FROM product 
      WHERE loginid = ?`,
      [userId],
    )

    if (services.length === 0) {
      // If no product record exists, create one with default services
      console.log(`⚠️ No product record found for user ID: ${userId}, creating default record`)

      const [userInfo] = await empCloudDbPool.execute("SELECT firstname, lastname FROM useraccounts WHERE id = ?", [
        userId,
      ])

      const productName =
        userInfo.length > 0 ? `${userInfo[0].firstname} ${userInfo[0].lastname} Organization` : "Organization"

      await empCloudDbPool.execute(
        `INSERT INTO product (loginid, productname, flag1, flag2, flag3, flag4, flag5, flag6, flag7, flag8, datecreated) 
         VALUES (?, ?, 1, 1, 0, 0, 0, 0, 0, 0, NOW())`,
        [userId, productName],
      )

      // Fetch the newly created record
      const [newServices] = await empCloudDbPool.execute(`SELECT * FROM product WHERE loginid = ?`, [userId])

      return res.json({
        success: true,
        data: newServices[0],
        message: "Default service configuration created",
      })
    }

    console.log(`✅ Found service configuration for user ID: ${userId}`)

    res.json({
      success: true,
      data: services[0],
    })
  } catch (error) {
    console.error("❌ Get services error:", error)
    res.status(500).json({
      success: false,
      message: "Failed to fetch services",
      error: process.env.NODE_ENV === "development" ? error.message : undefined,
    })
  }
}

// Update user's service configuration
const updateServices = async (req, res) => {
  try {
    const userId = req.user.userId
    const { productname, flag1, flag2, flag3, flag4, flag5, flag6, flag7, flag8 } = req.body
    const empCloudDbPool = req.app.locals.empCloudDbPool

    console.log(`🔄 Updating services for user ID: ${userId}`)

    // Validate flag values (should be 0 or 1)
    const flags = { flag1, flag2, flag3, flag4, flag5, flag6, flag7, flag8 }
    for (const [key, value] of Object.entries(flags)) {
      if (value !== undefined && value !== 0 && value !== 1) {
        return res.status(400).json({
          success: false,
          message: `Invalid value for ${key}. Must be 0 or 1.`,
        })
      }
    }

    // Check if product record exists
    const [existingServices] = await empCloudDbPool.execute("SELECT id FROM product WHERE loginid = ?", [userId])

    if (existingServices.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Service configuration not found",
      })
    }

    // Build dynamic update query
    const updateFields = []
    const updateValues = []

    // Only update productname if it's provided (make it optional)
    if (productname !== undefined && productname !== null) {
      updateFields.push("productname = ?")
      updateValues.push(productname)
    }

    // Add flag updates
    Object.entries(flags).forEach(([key, value]) => {
      if (value !== undefined) {
        updateFields.push(`${key} = ?`)
        updateValues.push(value)
      }
    })

    if (updateFields.length === 0) {
      return res.status(400).json({
        success: false,
        message: "No fields to update",
      })
    }

    // Add user ID for WHERE clause
    updateValues.push(userId)

    await empCloudDbPool.execute(`UPDATE product SET ${updateFields.join(", ")} WHERE loginid = ?`, updateValues)

    // Fetch updated record
    const [updatedServices] = await empCloudDbPool.execute("SELECT * FROM product WHERE loginid = ?", [userId])

    console.log(`✅ Services updated successfully for user ID: ${userId}`)

    res.json({
      success: true,
      message: "Services updated successfully",
      data: updatedServices[0],
    })
  } catch (error) {
    console.error("❌ Update services error:", error)
    res.status(500).json({
      success: false,
      message: "Failed to update services",
      error: process.env.NODE_ENV === "development" ? error.message : undefined,
    })
  }
}

// Get service status summary
const getServiceStatus = async (req, res) => {
  try {
    const userId = req.user.userId
    const empCloudDbPool = req.app.locals.empCloudDbPool

    const [services] = await empCloudDbPool.execute(
      "SELECT flag1, flag2, flag3, flag4, flag5, flag6, flag7, flag8 FROM product WHERE loginid = ?",
      [userId],
    )

    if (services.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Service configuration not found",
      })
    }

    const serviceData = services[0]
    const enabledServices = Object.values(serviceData).filter((flag) => flag === 1).length
    const totalServices = Object.keys(serviceData).length

    res.json({
      success: true,
      data: {
        enabledServices,
        totalServices,
        services: serviceData,
      },
    })
  } catch (error) {
    console.error("❌ Get service status error:", error)
    res.status(500).json({
      success: false,
      message: "Failed to fetch service status",
      error: process.env.NODE_ENV === "development" ? error.message : undefined,
    })
  }
}

module.exports = {
  getServices,
  updateServices,
  getServiceStatus,
}
