const validateLoginId = (req, res, next) => {
  const { loginid } = req.params;
  
  if (!loginid || isNaN(parseInt(loginid))) {
    return res.status(400).json({
      success: false,
      message: 'Invalid login ID',
      error: 'Login ID must be a valid number'
    });
  }
  
  // Convert to integer for consistency
  req.params.loginid = parseInt(loginid);
  next();
};

const validateUserId = (req, res, next) => {
  const { userid } = req.params;
  
  if (!userid || isNaN(parseInt(userid))) {
    return res.status(400).json({
      success: false,
      message: 'Invalid user ID',
      error: 'User ID must be a valid number'
    });
  }
  
  // Convert to integer for consistency
  req.params.userid = parseInt(userid);
  next();
};

const validateFlags = (req, res, next) => {
  const validFlags = ['flag1', 'flag2', 'flag3', 'flag4', 'flag5', 'flag6', 'flag7', 'flag8'];
  const requestData = req.body.flags || req.body;
  const requestFlags = Object.keys(requestData).filter(key => key.startsWith('flag'));
  
  const invalidFlags = requestFlags.filter(flag => !validFlags.includes(flag));
  
  if (invalidFlags.length > 0) {
    return res.status(400).json({
      success: false,
      message: 'Invalid flags provided',
      error: `Invalid flags: ${invalidFlags.join(', ')}`
    });
  }
  
  // Check if flag values are valid (0, 1, true, false)
  for (const flag of requestFlags) {
    const value = requestData[flag];
    if (value !== 0 && value !== 1 && value !== true && value !== false) {
      return res.status(400).json({
        success: false,
        message: 'Invalid flag value',
        error: `Flag ${flag} must be 0, 1, true, or false`
      });
    }
  }
  
  next();
};

const validateDateRange = (req, res, next) => {
  const { startDate, endDate } = req.query;
  
  if (startDate && !isValidDate(startDate)) {
    return res.status(400).json({
      success: false,
      message: 'Invalid start date',
      error: 'Start date must be in YYYY-MM-DD format'
    });
  }
  
  if (endDate && !isValidDate(endDate)) {
    return res.status(400).json({
      success: false,
      message: 'Invalid end date',
      error: 'End date must be in YYYY-MM-DD format'
    });
  }
  
  if (startDate && endDate && new Date(startDate) > new Date(endDate)) {
    return res.status(400).json({
      success: false,
      message: 'Invalid date range',
      error: 'Start date must be before end date'
    });
  }
  
  next();
};

const validatePagination = (req, res, next) => {
  const { limit, offset } = req.query;
  
  if (limit && (isNaN(parseInt(limit)) || parseInt(limit) < 1 || parseInt(limit) > 1000)) {
    return res.status(400).json({
      success: false,
      message: 'Invalid limit',
      error: 'Limit must be a number between 1 and 1000'
    });
  }
  
  if (offset && (isNaN(parseInt(offset)) || parseInt(offset) < 0)) {
    return res.status(400).json({
      success: false,
      message: 'Invalid offset',
      error: 'Offset must be a non-negative number'
    });
  }
  
  next();
};

const isValidDate = (dateString) => {
  const regex = /^\d{4}-\d{2}-\d{2}$/;
  if (!regex.test(dateString)) return false;
  
  const date = new Date(dateString);
  return date instanceof Date && !isNaN(date);
};

module.exports = {
  validateLoginId,
  validateUserId,
  validateFlags,
  validateDateRange,
  validatePagination
};