const mysql = require("mysql2/promise")

class DatabaseConfig {
  constructor() {
    this.empCloudDbPool = null
    this.adminDbPools = new Map()
  }

  /* ================= CREATE MAIN DB POOL (empcloud) ================= */
  createPools() {
    try {
      this.empCloudDbPool = mysql.createPool({
        host: process.env.DB_HOST || "localhost",
        user: process.env.DB_USER || "root",          // emp_root
        password: process.env.DB_PASSWORD || "",
        database: process.env.DB_NAME || "empcloud",
        waitForConnections: true,
        connectionLimit: 10,
        queueLimit: 0,
        connectTimeout: 60000,
        enableKeepAlive: true,
      })

      console.log("✅ EmpCloud database pool created")
      return { empCloudDbPool: this.empCloudDbPool }
    } catch (error) {
      console.error("❌ Error creating empCloud DB pool:", error)
      throw error
    }
  }

  /* ================= TEST ONLY MAIN DB ================= */
  async testConnection() {
    try {
      if (!this.empCloudDbPool) {
        throw new Error("EmpCloud DB pool not initialized")
      }

      await this.empCloudDbPool.execute("SELECT 1")
      console.log("✅ EmpCloud database connection successful")
      return true
    } catch (error) {
      console.error("❌ EmpCloud database connection failed:", error.message)
      throw error
    }
  }

  /* ================= LAZY ADMIN DB (edbX) ================= */
  getAdminDbPool(adminId) {
    const dbName = `edb${adminId}`

    if (!this.adminDbPools.has(dbName)) {
      const pool = mysql.createPool({
        host: process.env.DB_HOST || "localhost",
        user: process.env.DB_EMP_USER || process.env.DB_USER,          // empuser
        password: process.env.DB_EMP_PASSWORD || process.env.DB_PASSWORD,
        database: dbName,
        waitForConnections: true,
        connectionLimit: 5,
        queueLimit: 0,
        connectTimeout: 60000,
        enableKeepAlive: true,
      })

      this.adminDbPools.set(dbName, pool)
      console.log(`🧠 Admin DB pool created lazily for ${dbName}`)
    }

    return this.adminDbPools.get(dbName)
  }

  /* ================= CHECK ADMIN DB EXISTS & ACCESSIBLE ================= */
async testAdminDatabase(adminId) {
  const dbName = `edb${adminId}`

  try {
    const pool = this.getAdminDbPool(adminId)
    await pool.execute("SELECT 1")
    return true
  } catch (error) {
    // ✅ Database does not exist yet (EXPECTED)
    if (error.code === "ER_BAD_DB_ERROR") {
      // silently ignore – DB will be created by .NET later
      return false
    }

    // ⚠️ Permission or other real issue
    if (error.code === "ER_DBACCESS_DENIED_ERROR") {
      console.warn(`⚠️ Access denied to admin DB ${dbName}`)
      return false
    }

    // ❌ Unexpected DB error
    console.error(`❌ Admin DB ${dbName} error:`, error)
    return false
  }
}

  /* ================= CLOSE ALL POOLS ================= */
  async closeAllPools() {
    try {
      if (this.empCloudDbPool) {
        await this.empCloudDbPool.end()
        console.log("📊 EmpCloud DB pool closed")
      }

      for (const [dbName, pool] of this.adminDbPools) {
        await pool.end()
        console.log(`📊 ${dbName} DB pool closed`)
      }

      this.adminDbPools.clear()
    } catch (error) {
      console.error("❌ Error closing DB pools:", error)
    }
  }
}

module.exports = new DatabaseConfig()
