const bcrypt = require("bcryptjs")
const jwt = require("jsonwebtoken")

const transporter = require("../utils/mailer");
const { generateToken } = require("../utils/token");

class AuthController {
  constructor() {
    this.signup = this.signup.bind(this)
    this.login = this.login.bind(this)
    this.getProfile = this.getProfile.bind(this)
    this.updateProfile = this.updateProfile.bind(this)
    this.logout = this.logout.bind(this)

    this.verifyEmail = this.verifyEmail.bind(this)
    this.resendVerificationEmail = this.resendVerificationEmail.bind(this)
    this.forgotPassword = this.forgotPassword.bind(this)
    this.resetPassword = this.resetPassword.bind(this)
  }


  // Generate organization ID using the user's database ID
  generateOrgId(userId) {
    return `org${userId}`
  }

  async signup(req, res) {
    const { firstname, lastname, email, username, password } = req.body;

    try {
      // Validation
      if (!firstname || !lastname || !email || !username || !password) {
        return res.status(400).json({
          success: false,
          message: "All fields are required",
        });
      }

      // Email validation
      const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
      if (!emailRegex.test(email)) {
        return res.status(400).json({
          success: false,
          message: "Invalid email format",
        });
      }

      // Password validation
      if (password.length < 6) {
        return res.status(400).json({
          success: false,
          message: "Password must be at least 6 characters long",
        });
      }

      const empCloudDbPool = req.app.locals.empCloudDbPool;
      if (!empCloudDbPool) {
        console.error("ERROR: empCloudDbPool not available on req.app.locals");
        return res.status(500).json({ success: false, message: "Database not initialized" });
      }

      // Check if user already exists
      const [existingUsers] = await empCloudDbPool.execute(
        "SELECT id FROM useraccounts WHERE email = ? OR username = ?",
        [email, username]
      );

      if (existingUsers.length > 0) {
        return res.status(400).json({
          success: false,
          message: "User with this email or username already exists",
        });
      }

      // Hash password
      const saltRounds = 12;
      const hashedPassword = await bcrypt.hash(password, saltRounds);

      // 1) Create user without orgid (we will update it after insert)
      const [result] = await empCloudDbPool.execute(
        `INSERT INTO useraccounts
(firstname, lastname, email, username, password, actype, datecreated)
VALUES (?, ?, ?, ?, ?, 1, NOW())`,
        [firstname, lastname, email, username, hashedPassword]
      );

      const userId = result.insertId;
      console.log("DEBUG: user created with insertId =", userId);
      // ================= EMAIL VERIFICATION =================
      const verifyToken = generateToken();
      const verifyExpiry = new Date(Date.now() + 24 * 60 * 60 * 1000); // 24 hours

      await empCloudDbPool.execute(
        `UPDATE useraccounts
   SET email_verify_token = ?, email_verify_expires = ?
   WHERE id = ?`,
        [verifyToken, verifyExpiry, userId]
      );

      // Send verification email
      const verifyLink = `${process.env.FRONTEND_URL}/app2/verify-email/${verifyToken}`;

      await transporter.sendMail({
        from: `"WorkDesQ Security" <${process.env.EMAIL_USER}>`,
        to: email,
        subject: "Verify your email address – WorkDesQ",
        html: `
  <div style="background-color:#f4f7fb;padding:40px 10px;">
    <table width="100%" cellpadding="0" cellspacing="0" style="max-width:600px;margin:auto;background:#ffffff;border-radius:12px;box-shadow:0 8px 30px rgba(0,0,0,0.08);overflow:hidden;font-family:Segoe UI,Roboto,Helvetica,Arial,sans-serif;">
      
      <!-- HEADER -->
      <tr>
        <td style="background:linear-gradient(135deg,#2563eb,#4f46e5);padding:28px 32px;color:#ffffff;">
          <h1 style="margin:0;font-size:22px;font-weight:600;">WorkDesQ</h1>
          <p style="margin:6px 0 0;font-size:14px;opacity:0.9;">
            Secure workforce monitoring platform
          </p>
        </td>
      </tr>

      <!-- BODY -->
      <tr>
        <td style="padding:32px;color:#1f2937;">
          <h2 style="font-size:20px;margin:0 0 12px;font-weight:600;">
            Verify your email address
          </h2>

          <p style="font-size:14px;line-height:1.7;color:#4b5563;margin:0 0 20px;">
            Hi <strong>${firstname}</strong>,<br><br>
            Thank you for creating your WorkDesQ account.  
            To activate your account and keep it secure, please verify your email address.
          </p>

          <!-- BUTTON -->
          <div style="text-align:center;margin:30px 0;">
            <a href="${verifyLink}"
              style="background:linear-gradient(135deg,#2563eb,#4f46e5);
                     color:#ffffff;
                     padding:14px 32px;
                     text-decoration:none;
                     font-size:15px;
                     font-weight:600;
                     border-radius:8px;
                     display:inline-block;">
              Verify Email Address
            </a>
          </div>

          <p style="font-size:13px;color:#6b7280;line-height:1.6;">
            This verification link will expire in <strong>24 hours</strong>.  
            If you didn’t create this account, you can safely ignore this email.
          </p>

          <!-- SECURITY NOTE -->
          <div style="margin-top:24px;padding:14px;border-radius:8px;background:#f9fafb;border:1px solid #e5e7eb;">
            <p style="margin:0;font-size:12px;color:#6b7280;">
              🔒 For your security, never share this link with anyone.
            </p>
          </div>
        </td>
      </tr>

      <!-- FOOTER -->
      <tr>
        <td style="padding:20px 32px;text-align:center;font-size:12px;color:#9ca3af;background:#f9fafb;">
          © ${new Date().getFullYear()} WorkDesQ. All rights reserved.<br/>
          This email was sent because you signed up for a WorkDesQ account.
        </td>
      </tr>
    </table>
  </div>
  `,
      });



      // 2) Generate and update orgId for the user
      const orgId = this.generateOrgId(userId);
      await empCloudDbPool.execute("UPDATE useraccounts SET orgid = ? WHERE id = ?", [orgId, userId]);
      console.log(`✅ Updated user ${userId} with orgId: ${orgId}`);

      // 3) Create product record (capture productId)
      let productId = null;
      try {
        const productName = `${firstname} ${lastname} Organization`;
        const [productRes] = await empCloudDbPool.execute(
          `INSERT INTO product (loginid, productname, flag1, flag2, flag3, flag4, flag5, flag6, flag7, flag8, datecreated)
         VALUES (?, ?, 1, 1, 0, 0, 0, 0, 0, 0, NOW())`,
          [userId, productName]
        );
        productId = productRes.insertId;
        console.log(`✅ Product record created for user ID: ${userId} with productId: ${productId}`);
      } catch (productError) {
        console.error(`⚠️ Failed to create product record for user ID: ${userId}:`, productError);
        // If product creation fails and productId remains null, license insert may fail if DB requires productid NOT NULL.
        // Decide whether to abort signup or proceed without product. Here we proceed but will log productId=null.
      }

      // 4) Insert license row using productId (preferred — keeps referential integrity)
      try {
        const licenseSql = `
        INSERT INTO license
          (loginid, productid, nouser, remaininguser, tothdspace, status, licdays, curhdspace, hdfullpercentage, datecreated)
        VALUES (?, ?, ?, ?, ?, ?, ?, NULL, ?, NOW())
      `;
        const licenseParams = [
          userId,            // loginid
          productId,         // productid (if null, DB must allow NULL)
          5,                 // nouser - default demo employees
          5,                 // remaininguser - initially same as nouser
          1024,              // tothdspace in MB (1GB = 1024 MB)
          1,                 // status = 1 (active)
          30,                // licdays = 30 days demo
          0.0                // hdfullpercentage
        ];

        console.log("DEBUG: about to insert license with params:", licenseParams);
        const [licResult] = await empCloudDbPool.execute(licenseSql, licenseParams);
        console.log("DEBUG: license insert result =", licResult);
        if (licResult.affectedRows === 1) {
          console.log(`✅ License created for userId=${userId}, productId=${productId}, licenseId=${licResult.insertId || "(none)"}`);
        } else {
          console.warn("WARN: license insert did not report affectedRows === 1", licResult);
        }
      } catch (licErr) {
        console.error("❌ Error inserting license (full):", licErr);
        // Continue signup even if license insert fails — you may choose to rollback if you prefer atomic behavior.
      }

      // 5) Call .NET API to initialize system for user (unchanged)
      let dotNetApiResult = { status: "PENDING", message: "API call not attempted" };
      try {
        const fetch = require("node-fetch");
        const dotNetResponse = await fetch(`https://api.workdesq.ai/api/Home/init?param=${userId}`, {
          method: "GET",
          headers: { "Content-Type": "application/json", "User-Agent": "Employee-Monitoring-System/1.0" },
          timeout: 15000,
        });
        const responseText = await dotNetResponse.text();

        if (dotNetResponse.ok) {
          dotNetApiResult = {
            status: "OK",
            message: "System initialized successfully",
            response: responseText,
            apiUrl: `https://api.workdesq.ai/api/Home/init?param=${userId}`,
            userId,
          };
          console.log("✅ .NET API call successful - System initialized");
        } else {
          dotNetApiResult = {
            status: "ERROR",
            message: `API returned status ${dotNetResponse.status}`,
            error: responseText,
            apiUrl: `https://api.workdesq.ai/api/Home/init?param=${userId}`,
            userId,
          };
          console.error("❌ .NET API call failed:", dotNetResponse.status, responseText);
        }
      } catch (dotNetError) {
        console.error("❌ .NET API call error:", dotNetError);
        dotNetApiResult = {
          status: "ERROR",
          message: "Network error calling .NET API",
          error: dotNetError.message,
          apiUrl: `https://api.workdesq.ai/api/Home/init?param=${userId}`,
          userId,
        };
      }

      // 6) Final response
      res.status(201).json({
        success: true,
        message: "User created successfully",
        data: {
          id: userId,
          firstname,
          lastname,
          email,
          username,
          orgId,
          productId,
          dotNetApi: dotNetApiResult,
        },
      });

    } catch (error) {
      console.error("❌ Signup error:", error);
      res.status(500).json({
        success: false,
        message: "Internal server error",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      });
    }
  }


  async login(req, res) {
    const { username, password } = req.body

    try {
      // Validation
      if (!username || !password) {
        return res.status(400).json({
          success: false,
          message: "Username and password are required",
        })
      }

      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Find user by username or email
      const [users] = await empCloudDbPool.execute("SELECT * FROM useraccounts WHERE username = ? OR email = ?", [
        username,
        username,
      ])

      if (users.length === 0) {
        return res.status(401).json({
          success: false,
          message: "Invalid credentials",
        })
      }

      const user = users[0]
      if (!user.email_verified) {
        return res.status(403).json({
          success: false,
          message: "Please verify your email before logging in",
        });
      }


      // Verify password
      const isPasswordValid = await bcrypt.compare(password, user.password)
      if (!isPasswordValid) {
        return res.status(401).json({
          success: false,
          message: "Invalid credentials",
        })
      }

      // Generate JWT token
      const token = jwt.sign(
        {
          userId: user.id,
          username: user.username,
          orgId: user.orgid,
          actype: user.actype,
        },
        process.env.JWT_SECRET || "your-secret-key",
        { expiresIn: "24h" },
      )

      // Remove password from user object
      const { password: _, ...userWithoutPassword } = user

      console.log(`✅ User logged in successfully: ${user.username} (ID: ${user.id})`)

      res.json({
        success: true,
        message: "Login successful",
        token,
        user: {
          id: user.id,
          firstname: user.firstname,
          lastname: user.lastname,
          email: user.email,
          username: user.username,
          orgid: user.orgid,
          actype: user.actype
        },
      })
    } catch (error) {
      console.error("❌ Login error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  async getProfile(req, res) {
    try {
      const empCloudDbPool = req.app.locals.empCloudDbPool

      const [users] = await empCloudDbPool.execute(
        "SELECT id, firstname, lastname, email, username, orgid, link, datecreated FROM useraccounts WHERE id = ?",
        [req.user.userId],
      )

      if (users.length === 0) {
        return res.status(404).json({
          success: false,
          message: "User not found",
        })
      }

      res.json({
        success: true,
        user: users[0],
      })
    } catch (error) {
      console.error("Get profile error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  async updateProfile(req, res) {
    const { firstname, lastname, email } = req.body

    try {
      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Check if email is already taken by another user
      if (email) {
        const [existingUsers] = await empCloudDbPool.execute(
          "SELECT id FROM useraccounts WHERE email = ? AND id != ?",
          [email, req.user.userId],
        )

        if (existingUsers.length > 0) {
          return res.status(400).json({
            success: false,
            message: "Email is already taken",
          })
        }
      }

      // Update user profile
      await empCloudDbPool.execute("UPDATE useraccounts SET firstname = ?, lastname = ?, email = ? WHERE id = ?", [
        firstname,
        lastname,
        email,
        req.user.userId,
      ])

      res.json({
        success: true,
        message: "Profile updated successfully",
      })
    } catch (error) {
      console.error("Update profile error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  // Method to update the .exe file link (can be called separately when needed)
  async updateExeLink(req, res) {
    const { exeLink } = req.body

    try {
      if (!exeLink) {
        return res.status(400).json({
          success: false,
          message: "Exe link is required",
        })
      }

      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Update the link field with the .exe file link
      await empCloudDbPool.execute("UPDATE useraccounts SET link = ? WHERE id = ?", [exeLink, req.user.userId])

      console.log(`✅ Exe link updated for user ID: ${req.user.userId}`)

      res.json({
        success: true,
        message: "Exe link updated successfully",
        data: {
          userId: req.user.userId,
          exeLink: exeLink,
        },
      })
    } catch (error) {
      console.error("❌ Update exe link error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  async logout(req, res) {
    res.json({
      success: true,
      message: "Logged out successfully",
    })
  }
  async verifyEmail(req, res) {
    const { token } = req.params;
    const empCloudDbPool = req.app.locals.empCloudDbPool;

    // 1️⃣ Check valid token (not expired)
    const [users] = await empCloudDbPool.execute(
      `SELECT id, email_verified
     FROM useraccounts
     WHERE email_verify_token = ?
     AND email_verify_expires > NOW()`,
      [token]
    );

    // 2️⃣ Token not found / expired
    if (!users.length) {
      return res.status(400).json({
        success: false,
        message: "Invalid or expired verification link",
      });
    }

    // 3️⃣ If already verified (edge case)
    if (users[0].email_verified) {
      return res.json({
        success: true,
        message: "Email already verified",
      });
    }

    // 4️⃣ Mark as verified
    await empCloudDbPool.execute(
      `UPDATE useraccounts
     SET email_verified = 1,
         email_verify_token = NULL,
         email_verify_expires = NULL
     WHERE id = ?`,
      [users[0].id]
    );

    return res.json({
      success: true,
      message: "Email verified successfully",
    });
  }




  async resendVerificationEmail(req, res) {
    try {
      const { email } = req.body;
      const empCloudDbPool = req.app.locals.empCloudDbPool;

      if (!email) {
        return res.status(400).json({
          success: false,
          message: "Email or username is required",
        });
      }

      // 🔍 Find user by EMAIL or USERNAME
      const [users] = await empCloudDbPool.execute(
        `SELECT id, firstname, email, email_verified
       FROM useraccounts
       WHERE email = ? OR username = ?`,
        [email, email]
      );

      // 🔒 Do NOT reveal user existence
      if (!users.length) {
        return res.json({
          success: true,
          message: "If this email exists, a verification link has been sent",
        });
      }

      const user = users[0];

      // ✅ Already verified
      if (user.email_verified) {
        return res.json({
          success: true,
          message: "Email is already verified",
        });
      }

      // 🔑 Generate NEW token
      const verifyToken = generateToken();
      const verifyExpiry = new Date(Date.now() + 24 * 60 * 60 * 1000); // 24 hours

      await empCloudDbPool.execute(
        `UPDATE useraccounts
       SET email_verify_token = ?, email_verify_expires = ?
       WHERE id = ?`,
        [verifyToken, verifyExpiry, user.id]
      );

      const verifyLink = `${process.env.FRONTEND_URL}/app2/verify-email/${verifyToken}`;

      // 📧 SEND PREMIUM EMAIL
      await transporter.sendMail({
        from: `"WorkDesQ Security" <${process.env.EMAIL_USER}>`,
        to: user.email, // ✅ always real email
        subject: "Verify your email address – WorkDesQ",
        html: `
      <div style="background-color:#f4f7fb;padding:40px 10px;">
        <table width="100%" cellpadding="0" cellspacing="0"
          style="max-width:600px;margin:auto;background:#ffffff;
                 border-radius:12px;box-shadow:0 8px 30px rgba(0,0,0,0.08);
                 overflow:hidden;font-family:Segoe UI,Roboto,Helvetica,Arial,sans-serif;">

          <!-- HEADER -->
          <tr>
            <td style="background:linear-gradient(135deg,#2563eb,#4f46e5);
                       padding:28px 32px;color:#ffffff;">
              <h1 style="margin:0;font-size:22px;font-weight:600;">WorkDesQ</h1>
              <p style="margin:6px 0 0;font-size:14px;opacity:0.9;">
                Secure workforce monitoring platform
              </p>
            </td>
          </tr>

          <!-- BODY -->
          <tr>
            <td style="padding:32px;color:#1f2937;">
              <h2 style="font-size:20px;margin:0 0 12px;font-weight:600;">
                Verify your email address
              </h2>

              <p style="font-size:14px;line-height:1.7;color:#4b5563;margin:0 0 20px;">
                Hi <strong>${user.firstname}</strong>,<br><br>
                You recently requested a new verification link for your WorkDesQ account.
                Please confirm your email address by clicking the button below.
              </p>

              <!-- BUTTON -->
              <div style="text-align:center;margin:30px 0;">
                <a href="${verifyLink}"
                  style="background:linear-gradient(135deg,#2563eb,#4f46e5);
                         color:#ffffff;
                         padding:14px 32px;
                         text-decoration:none;
                         font-size:15px;
                         font-weight:600;
                         border-radius:8px;
                         display:inline-block;">
                  Verify Email Address
                </a>
              </div>

              <p style="font-size:13px;color:#6b7280;line-height:1.6;">
                This verification link will expire in <strong>24 hours</strong>.
                If you didn’t request this email, you can safely ignore it.
              </p>

              <div style="margin-top:24px;padding:14px;border-radius:8px;
                          background:#f9fafb;border:1px solid #e5e7eb;">
                <p style="margin:0;font-size:12px;color:#6b7280;">
                  🔒 For your security, never share this link with anyone.
                </p>
              </div>
            </td>
          </tr>

          <!-- FOOTER -->
          <tr>
            <td style="padding:20px 32px;text-align:center;
                       font-size:12px;color:#9ca3af;background:#f9fafb;">
              © ${new Date().getFullYear()} WorkDesQ. All rights reserved.<br/>
              This email was sent because a verification was requested for your account.
            </td>
          </tr>
        </table>
      </div>
      `,
      });

      return res.json({
        success: true,
        message: "Verification email resent successfully",
      });
    } catch (error) {
      console.error("❌ Resend verification error:", error);
      return res.status(500).json({
        success: false,
        message: "Something went wrong. Please try again later.",
      });
    }
  }



  async forgotPassword(req, res) {
    try {
      const { email } = req.body;
      const empCloudDbPool = req.app.locals.empCloudDbPool;

      if (!email) {
        return res.status(400).json({
          success: false,
          message: "Email or username is required",
        });
      }

      // Find user by email OR username
      const [users] = await empCloudDbPool.execute(
        `SELECT id, firstname, email
FROM useraccounts
WHERE (email = ? OR username = ?)
AND email_verified = 1`,
        [email, email]
      );

      // Security: do not reveal user existence
      if (!users.length) {
        return res.json({
          success: true,
          message: "If this account exists, a reset link has been sent",
        });
      }

      const user = users[0];

      // Generate reset token
      const resetToken = generateToken();
      const resetExpiry = new Date(Date.now() + 60 * 60 * 1000); // 1 hour

      await empCloudDbPool.execute(
        `UPDATE useraccounts
       SET reset_password_token = ?, reset_password_expires = ?
       WHERE id = ?`,
        [resetToken, resetExpiry, user.id]
      );

      const resetLink = `${process.env.FRONTEND_URL}/app2/reset-password/${resetToken}`;

      // Send premium reset email
      await transporter.sendMail({
        from: `"WorkDesQ Security" <${process.env.EMAIL_USER}>`,
        to: user.email,
        subject: "Reset your password – WorkDesQ",
        html: `
     <div style="background:#f4f7fb;padding:40px 12px;">
  <table width="100%" style="max-width:600px;margin:auto;background:#ffffff;
    border-radius:14px;box-shadow:0 12px 30px rgba(0,0,0,.08);
    font-family:Segoe UI,Roboto,Arial,sans-serif;overflow:hidden;">

    <tr>
      <td style="padding:28px 32px;background:linear-gradient(135deg,#2563eb,#4f46e5);color:white;">
        <h1 style="margin:0;font-size:22px;">WorkDesQ</h1>
        <p style="margin:6px 0 0;font-size:14px;opacity:.9">
          Password Recovery
        </p>
      </td>
    </tr>

    <tr>
      <td style="padding:32px;color:#1f2937;">
        <h2 style="margin:0 0 12px;font-size:20px;">
          Reset your password
        </h2>

        <p style="font-size:14px;line-height:1.7;color:#4b5563;">
          Hi <strong>${user.firstname}</strong>,<br><br>
          We received a request to reset your password.
          Click the button below to continue.
        </p>

        <div style="text-align:center;margin:28px 0;">
          <a href="${resetLink}" style="
            background:linear-gradient(135deg,#2563eb,#4f46e5);
            color:white;text-decoration:none;
            padding:14px 36px;border-radius:8px;
            font-weight:600;font-size:15px;">
            Reset Password
          </a>
        </div>

        <p style="font-size:13px;color:#6b7280;">
          This link expires in <strong>1 hour</strong>.  
          If you didn’t request this, you can safely ignore this email.
        </p>

        <div style="margin-top:24px;padding:14px;border-radius:8px;
          background:#f9fafb;border:1px solid #e5e7eb;">
          <p style="margin:0;font-size:12px;color:#6b7280;">
            🔒 Never share this link with anyone.
          </p>
        </div>
      </td>
    </tr>

    <tr>
      <td style="padding:20px;text-align:center;font-size:12px;color:#9ca3af;">
        © ${new Date().getFullYear()} WorkDesQ · Security Team
      </td>
    </tr>
  </table>
</div>

      `,
      });

      return res.json({
        success: true,
        message: "If this account exists, a reset link has been sent",
      });
    } catch (error) {
      console.error("❌ Forgot password error:", error);
      return res.status(500).json({
        success: false,
        message: "Something went wrong. Please try again later.",
      });
    }
  }
  async resetPassword(req, res) {
    try {
      const { token } = req.params;
      const { password } = req.body;
      const empCloudDbPool = req.app.locals.empCloudDbPool;

      if (!password || password.length < 6) {
        return res.status(400).json({
          success: false,
          message: "Password must be at least 6 characters long",
        });
      }

      const [users] = await empCloudDbPool.execute(
        `SELECT id
       FROM useraccounts
       WHERE reset_password_token = ?
       AND reset_password_expires > NOW()`,
        [token]
      );

      if (!users.length) {
        return res.status(400).json({
          success: false,
          message: "Invalid or expired reset link",
        });
      }

      const hashedPassword = await bcrypt.hash(password, 12);

      await empCloudDbPool.execute(
        `UPDATE useraccounts
       SET password = ?,
           reset_password_token = NULL,
           reset_password_expires = NULL
       WHERE id = ?`,
        [hashedPassword, users[0].id]
      );

      return res.json({
        success: true,
        message: "Password reset successfully",
      });
    } catch (error) {
      console.error("❌ Reset password error:", error);
      return res.status(500).json({
        success: false,
        message: "Something went wrong",
      });
    }
  }


  

}


module.exports = new AuthController()
