const mysql = require("mysql2/promise")

// Helper function to get user's database connection
const getUserDbConnection = async (userId) => {
  try {
    const dbname = `edb${userId}`

    console.log(`[v0] Connecting to database: ${dbname} for user ID: ${userId}`)

    // Create connection to user's specific database
    const userDbPool = mysql.createPool({
      host: process.env.DB_HOST || "localhost",
      user: process.env.DB_USER || "root",
      password: process.env.DB_PASSWORD || "",
      database: dbname,
      waitForConnections: true,
      connectionLimit: 10,
      queueLimit: 0,
    })

    return userDbPool
  } catch (error) {
    console.error("Error getting user database connection:", error)
    throw error
  }
}

const getDepartments = async (req, res) => {
  try {
    const userId = req.user.userId
    console.log(`[v0] Getting departments for user ID: ${userId}`)

    const userDbPool = await getUserDbConnection(userId)

    const [departments] = await userDbPool.execute("SELECT * FROM department ORDER BY id DESC")

    // Close the user database connection
    await userDbPool.end()

    res.json({
      success: true,
      message: "Departments retrieved successfully",
      data: departments,
    })
  } catch (error) {
    console.error("Error fetching departments:", error)
    res.status(500).json({
      success: false,
      message: "Failed to fetch departments",
      error: error.message,
    })
  }
}

const createDepartment = async (req, res) => {
  try {
    const { name } = req.body

    if (!name) {
      return res.status(400).json({
        success: false,
        message: "Department name is required",
      })
    }

    const userId = req.user.userId
    const userDbPool = await getUserDbConnection(userId)

    // Check if department already exists
    const [existingDept] = await userDbPool.execute("SELECT id FROM department WHERE name = ?", [name])

    if (existingDept.length > 0) {
      await userDbPool.end()
      return res.status(400).json({
        success: false,
        message: "Department with this name already exists",
      })
    }

    const [result] = await userDbPool.execute("INSERT INTO department (name) VALUES (?)", [name])

    // Get the created department
    const [newDepartment] = await userDbPool.execute("SELECT * FROM department WHERE id = ?", [result.insertId])

    // Close the user database connection
    await userDbPool.end()

    res.status(201).json({
      success: true,
      message: "Department created successfully",
      data: newDepartment[0],
    })
  } catch (error) {
    console.error("Error creating department:", error)
    res.status(500).json({
      success: false,
      message: "Failed to create department",
      error: error.message,
    })
  }
}

const updateDepartment = async (req, res) => {
  try {
    const { id } = req.params
    const { name } = req.body

    if (!name) {
      return res.status(400).json({
        success: false,
        message: "Department name is required",
      })
    }

    const userId = req.user.userId
    const userDbPool = await getUserDbConnection(userId)

    // Check if department exists
    const [existingDept] = await userDbPool.execute("SELECT id FROM department WHERE id = ?", [id])

    if (existingDept.length === 0) {
      await userDbPool.end()
      return res.status(404).json({
        success: false,
        message: "Department not found",
      })
    }

    // Check if another department with same name exists
    const [duplicateDept] = await userDbPool.execute("SELECT id FROM department WHERE name = ? AND id != ?", [name, id])

    if (duplicateDept.length > 0) {
      await userDbPool.end()
      return res.status(400).json({
        success: false,
        message: "Department with this name already exists",
      })
    }

    await userDbPool.execute("UPDATE department SET name = ? WHERE id = ?", [name, id])

    // Get updated department
    const [updatedDepartment] = await userDbPool.execute("SELECT * FROM department WHERE id = ?", [id])

    // Close the user database connection
    await userDbPool.end()

    res.json({
      success: true,
      message: "Department updated successfully",
      data: updatedDepartment[0],
    })
  } catch (error) {
    console.error("Error updating department:", error)
    res.status(500).json({
      success: false,
      message: "Failed to update department",
      error: error.message,
    })
  }
}

const deleteDepartment = async (req, res) => {
  try {
    const { id } = req.params
    const userId = req.user.userId
    const userDbPool = await getUserDbConnection(userId)

    // Check if department exists
    const [existingDept] = await userDbPool.execute("SELECT id FROM department WHERE id = ?", [id])

    if (existingDept.length === 0) {
      await userDbPool.end()
      return res.status(404).json({
        success: false,
        message: "Department not found",
      })
    }

    // Delete department
    await userDbPool.execute("DELETE FROM department WHERE id = ?", [id])

    // Close the user database connection
    await userDbPool.end()

    res.json({
      success: true,
      message: "Department deleted successfully",
    })
  } catch (error) {
    console.error("Error deleting department:", error)
    res.status(500).json({
      success: false,
      message: "Failed to delete department",
      error: error.message,
    })
  }
}

module.exports = {
  getDepartments,
  createDepartment,
  updateDepartment,
  deleteDepartment,
}
