const bcrypt = require("bcryptjs")
const jwt = require("jsonwebtoken")
const crypto = require("crypto")

class AuthController {
  constructor() {
    this.signup = this.signup.bind(this)
    this.login = this.login.bind(this)
    this.getProfile = this.getProfile.bind(this)
    this.updateProfile = this.updateProfile.bind(this)
    this.logout = this.logout.bind(this)
  }

  // Generate organization ID using the user's database ID
  generateOrgId(userId) {
    return `org${userId}`
  }

async signup(req, res) {
  const { firstname, lastname, email, username, password } = req.body;

  try {
    // Validation
    if (!firstname || !lastname || !email || !username || !password) {
      return res.status(400).json({
        success: false,
        message: "All fields are required",
      });
    }

    // Email validation
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return res.status(400).json({
        success: false,
        message: "Invalid email format",
      });
    }

    // Password validation
    if (password.length < 6) {
      return res.status(400).json({
        success: false,
        message: "Password must be at least 6 characters long",
      });
    }

    const empCloudDbPool = req.app.locals.empCloudDbPool;
    if (!empCloudDbPool) {
      console.error("ERROR: empCloudDbPool not available on req.app.locals");
      return res.status(500).json({ success: false, message: "Database not initialized" });
    }

    // Check if user already exists
    const [existingUsers] = await empCloudDbPool.execute(
      "SELECT id FROM useraccounts WHERE email = ? OR username = ?",
      [email, username]
    );

    if (existingUsers.length > 0) {
      return res.status(400).json({
        success: false,
        message: "User with this email or username already exists",
      });
    }

    // Hash password
    const saltRounds = 12;
    const hashedPassword = await bcrypt.hash(password, saltRounds);

    // 1) Create user without orgid (we will update it after insert)
    const [result] = await empCloudDbPool.execute(
      `INSERT INTO useraccounts
(firstname, lastname, email, username, password, actype, datecreated)
VALUES (?, ?, ?, ?, ?, 1, NOW())`,
      [firstname, lastname, email, username, hashedPassword]
    );

    const userId = result.insertId;
    console.log("DEBUG: user created with insertId =", userId);

    // 2) Generate and update orgId for the user
    const orgId = this.generateOrgId(userId);
    await empCloudDbPool.execute("UPDATE useraccounts SET orgid = ? WHERE id = ?", [orgId, userId]);
    console.log(`✅ Updated user ${userId} with orgId: ${orgId}`);

    // 3) Create product record (capture productId)
    let productId = null;
    try {
      const productName = `${firstname} ${lastname} Organization`;
      const [productRes] = await empCloudDbPool.execute(
        `INSERT INTO product (loginid, productname, flag1, flag2, flag3, flag4, flag5, flag6, flag7, flag8, datecreated)
         VALUES (?, ?, 1, 1, 0, 0, 0, 0, 0, 0, NOW())`,
        [userId, productName]
      );
      productId = productRes.insertId;
      console.log(`✅ Product record created for user ID: ${userId} with productId: ${productId}`);
    } catch (productError) {
      console.error(`⚠️ Failed to create product record for user ID: ${userId}:`, productError);
      // If product creation fails and productId remains null, license insert may fail if DB requires productid NOT NULL.
      // Decide whether to abort signup or proceed without product. Here we proceed but will log productId=null.
    }

    // 4) Insert license row using productId (preferred — keeps referential integrity)
    try {
      const licenseSql = `
        INSERT INTO license
          (loginid, productid, nouser, remaininguser, tothdspace, status, licdays, curhdspace, hdfullpercentage, datecreated)
        VALUES (?, ?, ?, ?, ?, ?, ?, NULL, ?, NOW())
      `;
      const licenseParams = [
        userId,            // loginid
        productId,         // productid (if null, DB must allow NULL)
        5,                 // nouser - default demo employees
        5,                 // remaininguser - initially same as nouser
        1024,              // tothdspace in MB (1GB = 1024 MB)
        1,                 // status = 1 (active)
        30,                // licdays = 30 days demo
        0.0                // hdfullpercentage
      ];

      console.log("DEBUG: about to insert license with params:", licenseParams);
      const [licResult] = await empCloudDbPool.execute(licenseSql, licenseParams);
      console.log("DEBUG: license insert result =", licResult);
      if (licResult.affectedRows === 1) {
        console.log(`✅ License created for userId=${userId}, productId=${productId}, licenseId=${licResult.insertId || "(none)"}`);
      } else {
        console.warn("WARN: license insert did not report affectedRows === 1", licResult);
      }
    } catch (licErr) {
      console.error("❌ Error inserting license (full):", licErr);
      // Continue signup even if license insert fails — you may choose to rollback if you prefer atomic behavior.
    }

    // 5) Call .NET API to initialize system for user (unchanged)
    let dotNetApiResult = { status: "PENDING", message: "API call not attempted" };
    try {
      const fetch = require("node-fetch");
      const dotNetResponse = await fetch(`http://103.148.165.134:5006/api/Home/init?param=${userId}`, {
        method: "GET",
        headers: { "Content-Type": "application/json", "User-Agent": "Employee-Monitoring-System/1.0" },
        timeout: 15000,
      });
      const responseText = await dotNetResponse.text();

      if (dotNetResponse.ok) {
        dotNetApiResult = {
          status: "OK",
          message: "System initialized successfully",
          response: responseText,
          apiUrl: `http://103.148.165.134:5006/api/Home/init?param=${userId}`,
          userId,
        };
        console.log("✅ .NET API call successful - System initialized");
      } else {
        dotNetApiResult = {
          status: "ERROR",
          message: `API returned status ${dotNetResponse.status}`,
          error: responseText,
          apiUrl: `http://103.148.165.134:5006/api/Home/init?param=${userId}`,
          userId,
        };
        console.error("❌ .NET API call failed:", dotNetResponse.status, responseText);
      }
    } catch (dotNetError) {
      console.error("❌ .NET API call error:", dotNetError);
      dotNetApiResult = {
        status: "ERROR",
        message: "Network error calling .NET API",
        error: dotNetError.message,
        apiUrl: `http://103.148.165.134:5006/api/Home/init?param=${userId}`,
        userId,
      };
    }

    // 6) Final response
    res.status(201).json({
      success: true,
      message: "User created successfully",
      data: {
        id: userId,
        firstname,
        lastname,
        email,
        username,
        orgId,
        productId,
        dotNetApi: dotNetApiResult,
      },
    });

  } catch (error) {
    console.error("❌ Signup error:", error);
    res.status(500).json({
      success: false,
      message: "Internal server error",
      error: process.env.NODE_ENV === "development" ? error.message : undefined,
    });
  }
}


  async login(req, res) {
    const { username, password } = req.body

    try {
      // Validation
      if (!username || !password) {
        return res.status(400).json({
          success: false,
          message: "Username and password are required",
        })
      }

      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Find user by username or email
      const [users] = await empCloudDbPool.execute("SELECT * FROM useraccounts WHERE username = ? OR email = ?", [
        username,
        username,
      ])

      if (users.length === 0) {
        return res.status(401).json({
          success: false,
          message: "Invalid credentials",
        })
      }

      const user = users[0]

      // Verify password
      const isPasswordValid = await bcrypt.compare(password, user.password)
      if (!isPasswordValid) {
        return res.status(401).json({
          success: false,
          message: "Invalid credentials",
        })
      }

      // Generate JWT token
      const token = jwt.sign(
        {
          userId: user.id,
          username: user.username,
          orgId: user.orgid,
          actype: user.actype,
        },
        process.env.JWT_SECRET || "your-secret-key",
        { expiresIn: "24h" },
      )

      // Remove password from user object
      const { password: _, ...userWithoutPassword } = user

      console.log(`✅ User logged in successfully: ${user.username} (ID: ${user.id})`)

      res.json({
         success: true,
  message: "Login successful",
  token,
  user: {
    id: user.id,
    firstname: user.firstname,
    lastname: user.lastname,
    email: user.email,
    username: user.username,
    orgid: user.orgid,
    actype: user.actype  
  },
      })
    } catch (error) {
      console.error("❌ Login error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  async getProfile(req, res) {
    try {
      const empCloudDbPool = req.app.locals.empCloudDbPool

      const [users] = await empCloudDbPool.execute(
        "SELECT id, firstname, lastname, email, username, orgid, link, datecreated FROM useraccounts WHERE id = ?",
        [req.user.userId],
      )

      if (users.length === 0) {
        return res.status(404).json({
          success: false,
          message: "User not found",
        })
      }

      res.json({
        success: true,
        user: users[0],
      })
    } catch (error) {
      console.error("Get profile error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  async updateProfile(req, res) {
    const { firstname, lastname, email } = req.body

    try {
      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Check if email is already taken by another user
      if (email) {
        const [existingUsers] = await empCloudDbPool.execute(
          "SELECT id FROM useraccounts WHERE email = ? AND id != ?",
          [email, req.user.userId],
        )

        if (existingUsers.length > 0) {
          return res.status(400).json({
            success: false,
            message: "Email is already taken",
          })
        }
      }

      // Update user profile
      await empCloudDbPool.execute("UPDATE useraccounts SET firstname = ?, lastname = ?, email = ? WHERE id = ?", [
        firstname,
        lastname,
        email,
        req.user.userId,
      ])

      res.json({
        success: true,
        message: "Profile updated successfully",
      })
    } catch (error) {
      console.error("Update profile error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  // Method to update the .exe file link (can be called separately when needed)
  async updateExeLink(req, res) {
    const { exeLink } = req.body

    try {
      if (!exeLink) {
        return res.status(400).json({
          success: false,
          message: "Exe link is required",
        })
      }

      const empCloudDbPool = req.app.locals.empCloudDbPool

      // Update the link field with the .exe file link
      await empCloudDbPool.execute("UPDATE useraccounts SET link = ? WHERE id = ?", [exeLink, req.user.userId])

      console.log(`✅ Exe link updated for user ID: ${req.user.userId}`)

      res.json({
        success: true,
        message: "Exe link updated successfully",
        data: {
          userId: req.user.userId,
          exeLink: exeLink,
        },
      })
    } catch (error) {
      console.error("❌ Update exe link error:", error)
      res.status(500).json({
        success: false,
        message: "Internal server error",
      })
    }
  }

  async logout(req, res) {
    res.json({
      success: true,
      message: "Logged out successfully",
    })
  }
}

module.exports = new AuthController()
