const databaseConfig = require("../config/database")

class LicenseController {
  async getLicenseInfo(req, res) {
    try {
      const empCloudDbPool = req.app.locals.empCloudDbPool
      const loginId = req.user.userId

      /* ================= FETCH LICENSE ================= */
      const [rows] = await empCloudDbPool.execute(
        `
        SELECT 
          l.tothdspace,
          l.curhdspace,
          l.licdays AS totalLicenseDays,
          l.hdfullpercentage,
          l.status,
          l.datecreated,
          l.nouser,
          l.remaininguser,
          ua.actype
        FROM license l
        JOIN useraccounts ua ON ua.id = l.loginid
        WHERE l.loginid = ?
        LIMIT 1
        `,
        [loginId]
      )

      if (!rows.length) {
        return res.status(404).json({
          success: false,
          message: "License not found",
        })
      }

      const lic = rows[0]

      /* ================= DATE CALC ================= */
      const today = new Date()
      const createdDate = new Date(lic.datecreated)

      const usedDays = Math.floor(
        (today.getTime() - createdDate.getTime()) /
          (1000 * 60 * 60 * 24)
      )

      let remainingDays = lic.totalLicenseDays - usedDays
      if (remainingDays < 0) remainingDays = 0

      /* ================= FLAGS ================= */
      const isExpired = lic.status === 2
      const isReadonly = isExpired

      /* ================= 🔒 HARD FREEZE USERS (ONCE ONLY) ================= */
      if (isExpired) {
        const adminDbPool = databaseConfig.getAdminDbPool(loginId)

        const [result] = await adminDbPool.execute(
          `
          UPDATE userdetails
          SET
            connected = 0,
            flag1 = 0,
            flag2 = 0,

            connchange = IF(connchange = 0, 1, connchange),
            f1change   = IF(f1change = 0, 1, f1change),
            f2change   = IF(f2change = 0, 1, f2change),

            setflag = 1,
            lastupdated = CURRENT_TIMESTAMP
          WHERE
            loginid = ?
            AND connected != 2
            AND (
              connected != 0
              OR f1change = 0
              OR f2change = 0
              OR connchange = 0
            )
          `,
          [loginId]
        )

        if (result.affectedRows > 0) {
          console.log(
            `🔒 License expired → users frozen once: ${result.affectedRows}`
          )
        }
      }

      /* ================= EXPIRY DATE ================= */
      const expiryDate = new Date(
        createdDate.getTime() +
          lic.totalLicenseDays * 24 * 60 * 60 * 1000
      )

      /* ================= RESPONSE ================= */
      res.json({
        success: true,
        data: {
          actype: lic.actype,
          isDemo: lic.actype === 1,
          isLicensed: lic.actype === 2,

          isExpired,
          isReadonly,

          nouser: lic.nouser,
          remaininguser: lic.remaininguser,

          totalHdSpace: lic.tothdspace,
          currentHdSpace: lic.curhdspace ?? 0,
          hdUsagePercentage: lic.hdfullpercentage,

          totalLicenseDays: lic.totalLicenseDays,
          usedDays,
          licenseDaysPending: remainingDays,

          expiryDate: expiryDate.toISOString().split("T")[0],
          status: isExpired ? "Expired" : "Active",
        },
      })
    } catch (error) {
      console.error("❌ License API error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to fetch license info",
      })
    }
  }
}

module.exports = new LicenseController()
