const databaseConfig = require("../config/database")

/* ================= LICENSE CHECK (empcloud) ================= */
async function checkLicenseActive(req) {
  const empCloudDbPool = req.app.locals.empCloudDbPool
  const adminId = req.user.userId

  const [[license]] = await empCloudDbPool.execute(
    "SELECT status FROM license WHERE loginid = ?",
    [adminId]
  )

  return license?.status === 2
}

/* ================= GET DEPARTMENTS (edbX) ================= */
const getDepartments = async (req, res) => {
  try {
    const adminId = req.user.userId
    console.log(`[v1] Getting departments for admin ID: ${adminId}`)

    // 1️⃣ Check if admin DB is ready
    const isAccessible = await databaseConfig.testAdminDatabase(adminId)
    if (!isAccessible) {
      return res.status(202).json({
        success: false,
        message: "Organization database is still initializing. Please try again shortly.",
      })
    }

    // 2️⃣ Use admin DB pool (empuser)
    const adminDbPool = databaseConfig.getAdminDbPool(adminId)

    const [departments] = await adminDbPool.execute(
      "SELECT * FROM department ORDER BY id DESC"
    )

    res.json({
      success: true,
      message: "Departments retrieved successfully",
      data: departments,
    })
  } catch (error) {
    console.error("Error fetching departments:", error)
    res.status(500).json({
      success: false,
      message: "Failed to fetch departments",
      error: error.message,
    })
  }
}

/* ================= CREATE DEPARTMENT (edbX) ================= */
const createDepartment = async (req, res) => {
  try {
    const { name } = req.body
    if (!name) {
      return res.status(400).json({
        success: false,
        message: "Department name is required",
      })
    }

    const isExpired = await checkLicenseActive(req)
    if (isExpired) {
      return res.status(403).json({
        success: false,
        message: "License expired. Department operations are not allowed.",
        code: "LICENSE_EXPIRED_READONLY",
      })
    }

    const adminId = req.user.userId
    const isAccessible = await databaseConfig.testAdminDatabase(adminId)
    if (!isAccessible) {
      return res.status(202).json({
        success: false,
        message: "Organization database is still initializing.",
      })
    }

    const adminDbPool = databaseConfig.getAdminDbPool(adminId)

    const [existingDept] = await adminDbPool.execute(
      "SELECT id FROM department WHERE name = ?",
      [name]
    )

    if (existingDept.length > 0) {
      return res.status(400).json({
        success: false,
        message: "Department with this name already exists",
      })
    }

    const [result] = await adminDbPool.execute(
      "INSERT INTO department (name) VALUES (?)",
      [name]
    )

    const [[newDepartment]] = await adminDbPool.execute(
      "SELECT * FROM department WHERE id = ?",
      [result.insertId]
    )

    res.status(201).json({
      success: true,
      message: "Department created successfully",
      data: newDepartment,
    })
  } catch (error) {
    console.error("Error creating department:", error)
    res.status(500).json({
      success: false,
      message: "Failed to create department",
      error: error.message,
    })
  }
}

/* ================= UPDATE DEPARTMENT (edbX) ================= */
const updateDepartment = async (req, res) => {
  try {
    const { id } = req.params
    const { name } = req.body

    if (!name) {
      return res.status(400).json({
        success: false,
        message: "Department name is required",
      })
    }

    const isExpired = await checkLicenseActive(req)
    if (isExpired) {
      return res.status(403).json({
        success: false,
        message: "License expired. Department operations are not allowed.",
        code: "LICENSE_EXPIRED_READONLY",
      })
    }

    const adminId = req.user.userId
    const isAccessible = await databaseConfig.testAdminDatabase(adminId)
    if (!isAccessible) {
      return res.status(202).json({
        success: false,
        message: "Organization database is still initializing.",
      })
    }

    const adminDbPool = databaseConfig.getAdminDbPool(adminId)

    const [existingDept] = await adminDbPool.execute(
      "SELECT id FROM department WHERE id = ?",
      [id]
    )

    if (existingDept.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Department not found",
      })
    }

    const [duplicateDept] = await adminDbPool.execute(
      "SELECT id FROM department WHERE name = ? AND id != ?",
      [name, id]
    )

    if (duplicateDept.length > 0) {
      return res.status(400).json({
        success: false,
        message: "Department with this name already exists",
      })
    }

    await adminDbPool.execute(
      "UPDATE department SET name = ? WHERE id = ?",
      [name, id]
    )

    const [[updatedDepartment]] = await adminDbPool.execute(
      "SELECT * FROM department WHERE id = ?",
      [id]
    )

    res.json({
      success: true,
      message: "Department updated successfully",
      data: updatedDepartment,
    })
  } catch (error) {
    console.error("Error updating department:", error)
    res.status(500).json({
      success: false,
      message: "Failed to update department",
      error: error.message,
    })
  }
}

/* ================= DELETE DEPARTMENT (edbX) ================= */
const deleteDepartment = async (req, res) => {
  try {
    const { id } = req.params

    const isExpired = await checkLicenseActive(req)
    if (isExpired) {
      return res.status(403).json({
        success: false,
        message: "License expired. Department operations are not allowed.",
        code: "LICENSE_EXPIRED_READONLY",
      })
    }

    const adminId = req.user.userId
    const isAccessible = await databaseConfig.testAdminDatabase(adminId)
    if (!isAccessible) {
      return res.status(202).json({
        success: false,
        message: "Organization database is still initializing.",
      })
    }

    const adminDbPool = databaseConfig.getAdminDbPool(adminId)

    const [existingDept] = await adminDbPool.execute(
      "SELECT id FROM department WHERE id = ?",
      [id]
    )

    if (existingDept.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Department not found",
      })
    }

    await adminDbPool.execute(
      "DELETE FROM department WHERE id = ?",
      [id]
    )

    res.json({
      success: true,
      message: "Department deleted successfully",
    })
  } catch (error) {
    console.error("Error deleting department:", error)
    res.status(500).json({
      success: false,
      message: "Failed to delete department",
      error: error.message,
    })
  }
}

module.exports = {
  getDepartments,
  createDepartment,
  updateDepartment,
  deleteDepartment,
}
