const databaseConfig = require("../config/database")
const JSZip = require("jszip")
const fetch = require("node-fetch")

async function checkLicenseActive(req) {
  const empCloudDbPool = req.app.locals.empCloudDbPool
  const adminId = req.user.userId

  const [[license]] = await empCloudDbPool.execute("SELECT status FROM license WHERE loginid = ?", [adminId])

  return license?.status === 2
}

class UserDetailsController {
  constructor() {
    this.getUserDetails = this.getUserDetails.bind(this)
    this.updateUserDetails = this.updateUserDetails.bind(this)
    this.bulkUpdateUserDetails = this.bulkUpdateUserDetails.bind(this)
    this.fetchUserImages = this.fetchUserImages.bind(this)
    this.getUserActivityData = this.getUserActivityData.bind(this)
    this.checkDataExists = this.checkDataExists.bind(this)
    this.downloadZipFile = this.downloadZipFile.bind(this)
    this.getUserActivityForReport = this.getUserActivityForReport.bind(this)
    this.getUserActivitySummary = this.getUserActivitySummary.bind(this)
    this.uninstallUser = this.uninstallUser.bind(this)
  }

  // ✅ Get all user details with email + department info + shift data
  async getUserDetails(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      console.log(`🔍 Fetching user details for admin ID: ${adminId}`)

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)
      const dbExists = await databaseConfig.testAdminDatabase(adminId)
      if (!dbExists) {
        return res.json({
          success: true,
          data: [],
          total: 0,
          message: `Admin database edb${adminId} is not accessible.`,
          database: `edb${adminId}`,
          accessible: false,
        })
      }

      const query = `
        SELECT 
          ud.id,
          ud.loginid,
          ud.userid,
          ul.email AS user_email,
          ud.devicename,
          ud.deviceid,
          ud.srnumber,
          ud.depid,
        COALESCE(d.name, 'Not Assigned') AS department_name,
          ud.flag1,
          ud.flag2,
          ud.flag3,
          ud.flag4,
          ud.flag5,
          ud.flag6,
          ud.flag7,
          ud.flag8,
          ud.active,
          ud.connected,
          ud.frequency,
          ud.setflag,
          ud.starttime,
          ud.endtime,
          ud.days,
          ud.username,
          ud.location,
          ud.os,
          ud.f1change,
          ud.f2change,
          ud.connchange,
          ud.freqchange,
          ud.shiftchange,
          ud.datecreated,
          ud.lastupdated
        FROM userdetails ud
        LEFT JOIN userlist ul 
          ON ul.id = ud.userid AND ul.loginid = ud.loginid
        LEFT JOIN department d
          ON d.id = ud.depid
        WHERE ud.loginid = ? AND ud.connected != 2
        ORDER BY ud.datecreated DESC
      `

      const [rows] = await adminDbPool.execute(query, [adminId])

      res.status(200).json({
        success: true,
        message: "User details retrieved successfully",
        data: rows,
        total: rows.length,
        database: `edb${adminId}`,
        accessible: true,
      })
    } catch (error) {
      console.error("❌ Get user details error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to retrieve user details",
      })
    }
  }

  // ✅ Update single user details (including shift data)
  async updateUserDetails(req, res) {
    try {
      const { id } = req.params
      const adminId = req.user.loginid || req.user.userId
      const updates = req.body

      console.log(`🔄 Updating user details ID: ${id} for admin: ${adminId}`)

    const isExpired = await checkLicenseActive(req)

if (isExpired) {
  // HARD STOP — nothing should touch flags
  return res.status(403).json({
    success: false,
    message: "License expired. Read-only mode.",
    code: "LICENSE_EXPIRED",
  })
}

      // 🔒 HARD BLOCK: License expired → NO department assign
      if (isExpired && updates.depid !== undefined) {
        return res.status(403).json({
          success: false,
          message: "License expired. Department assignment is not allowed.",
          code: "LICENSE_EXPIRED",
        })
      }

      // 🔒 Existing restriction (services / connection)
      if (isExpired && (updates.connected === 1 || updates.flag1 === 1 || updates.flag2 === 1)) {
        return res.status(403).json({
          success: false,
          message: "License expired. Please renew to continue services.",
          code: "LICENSE_EXPIRED",
        })
      }

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      const [currRows] = await adminDbPool.execute(
        `SELECT flag1, flag2, connected, f1change, f2change, connchange,
              frequency, starttime, endtime, days
       FROM userdetails WHERE id = ? AND loginid = ?`,
        [id, adminId],
      )

      if (currRows.length === 0) {
        return res.status(404).json({
          success: false,
          message: "User details not found or unauthorized",
        })
      }

      const current = currRows[0]

      // ✅ ONLY DEPARTMENT CHANGE CHECK
      const isOnlyDepartmentChange = Object.keys(updates).length === 1 && typeof updates.depid !== "undefined"

      if (current.f1change === 1) delete updates.f1change
      if (current.f2change === 1) delete updates.f2change
      if (current.connchange === 1) delete updates.connchange

      // Auto enable services on connect
      if (typeof updates.connected !== "undefined" && updates.connected === 1 && current.connected === 0) {
        updates.flag1 = 1
        updates.flag2 = 1
      }

      // Deactivate / Uninstall
      if (typeof updates.connected !== "undefined") {
        if (updates.connected === 0 || updates.connected === 2) {
          updates.flag1 = 0
          updates.flag2 = 0
        }
      }

      // ✅ COMPUTE CHANGE FLAGS (SAFE)
      const computedChanges = {}

      if (!isOnlyDepartmentChange) {
        if (typeof updates.flag1 !== "undefined" && updates.flag1 !== current.flag1) {
          computedChanges.f1change = 1
        }

        if (typeof updates.flag2 !== "undefined" && updates.flag2 !== current.flag2) {
          computedChanges.f2change = 1
        }

        if (typeof updates.connected !== "undefined" && updates.connected !== current.connected) {
          computedChanges.connchange = 1
        }

        if (typeof updates.frequency !== "undefined" && updates.frequency !== current.frequency) {
          computedChanges.freqchange = 1
        }
      }

      const shiftFieldsTouched =
        typeof updates.starttime !== "undefined" ||
        typeof updates.endtime !== "undefined" ||
        typeof updates.days !== "undefined"

      const shiftWillChange =
        (typeof updates.starttime !== "undefined" && updates.starttime !== current.starttime) ||
        (typeof updates.endtime !== "undefined" && updates.endtime !== current.endtime) ||
        (typeof updates.days !== "undefined" && updates.days !== current.days)

      if (shiftFieldsTouched) {
        computedChanges.shiftchange = shiftWillChange ? 1 : 0
      }

      // ✅ BUILD UPDATE QUERY
      const updateFields = []
      const updateValues = []

      Object.keys(updates).forEach((key) => {
        if (
          [
            "flag1",
            "flag2",
            "flag3",
            "flag4",
            "flag5",
            "flag6",
            "flag7",
            "flag8",
            "active",
            "connected",
            "frequency",
            "userid",
            "depid",
            "starttime",
            "endtime",
            "days",
          ].includes(key)
        ) {
          updateFields.push(`${key} = ?`)
          updateValues.push(updates[key])
        }
      })

      Object.keys(computedChanges).forEach((k) => {
        updateFields.push(`${k} = ?`)
        updateValues.push(computedChanges[k])
      })

      if (updateFields.length === 0) {
        return res.status(400).json({
          success: false,
          message: "No valid fields to update",
        })
      }

      updateFields.push(`setflag = 1`)
      updateValues.push(id, adminId)

      const query = `
      UPDATE userdetails
      SET ${updateFields.join(", ")}, lastupdated = CURRENT_TIMESTAMP
      WHERE id = ? AND loginid = ?
    `

      const [result] = await adminDbPool.execute(query, updateValues)

      if (result.affectedRows === 0) {
        return res.status(404).json({
          success: false,
          message: "User details not found or unauthorized",
        })
      }

      // 🔓 LICENSE UPDATE ON UNINSTALL
      if (typeof updates.connected !== "undefined" && updates.connected === 2 && current.connected !== 2) {
        const empCloudDbPool = req.app.locals.empCloudDbPool

        await empCloudDbPool.execute(
          `
        UPDATE license
        SET remaininguser = LEAST(remaininguser + 1, nouser)
        WHERE loginid = ?
        `,
          [adminId],
        )

        console.log("🔓 remaininguser updated due to uninstall")
      }

      res.status(200).json({
        success: true,
        message: "User details updated successfully",
      })
    } catch (error) {
      console.error("❌ Update user details error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to update user details",
      })
    }
  }

  // ✅ Bulk update user details (including shift data)
  async bulkUpdateUserDetails(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      const { updates } = req.body

      if (!Array.isArray(updates) || updates.length === 0) {
        return res.status(400).json({
          success: false,
          message: "Updates array is required",
        })
      }

    const isExpired = await checkLicenseActive(req)

if (isExpired) {
  return res.status(403).json({
    success: false,
    message: "License expired. Department operations are disabled.",
  })
}

      console.log(`🔄 Bulk updating ${updates.length} user details for admin: ${adminId}`)

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      let successCount = 0
      const errors = []

      for (const update of updates) {
        try {
          const { id, ...fields } = update

          const [currRows] = await adminDbPool.execute(
            `SELECT flag1, flag2, connected,
                  f1change, f2change, connchange,
                  frequency, starttime, endtime, days
           FROM userdetails
           WHERE id = ? AND loginid = ?`,
            [id, adminId],
          )

          if (currRows.length === 0) continue
          const current = currRows[0]

          const SERVICE_KEYS = ["flag1", "flag2", "flag3", "flag4", "flag5", "flag6", "flag7", "flag8"]

          const localUpdates = { ...fields }

          if (current.f1change === 1) delete localUpdates.f1change
          if (current.f2change === 1) delete localUpdates.f2change
          if (current.connchange === 1) delete localUpdates.connchange

          // Services can only change if user is active
          if (current.connected !== 1) {
            SERVICE_KEYS.forEach((k) => {
              if (k in localUpdates) delete localUpdates[k]
            })
          }

          // Deactivate
          if (typeof localUpdates.connected !== "undefined" && localUpdates.connected === 0) {
            localUpdates.flag1 = 0
            localUpdates.flag2 = 0
          }

          const computedChanges = {}

          if (
            typeof localUpdates.flag1 !== "undefined" &&
            localUpdates.flag1 !== current.flag1 &&
            current.f1change !== 1
          ) {
            computedChanges.f1change = 1
          }

          if (
            typeof localUpdates.flag2 !== "undefined" &&
            localUpdates.flag2 !== current.flag2 &&
            current.f2change !== 1
          ) {
            computedChanges.f2change = 1
          }

          if (
            typeof localUpdates.connected !== "undefined" &&
            localUpdates.connected !== current.connected &&
            current.connchange !== 1
          ) {
            computedChanges.connchange = 1
          }

          if (typeof localUpdates.frequency !== "undefined" && localUpdates.frequency !== current.frequency) {
            computedChanges.freqchange = 1
          }

          const shiftFieldsTouched =
            typeof localUpdates.starttime !== "undefined" ||
            typeof localUpdates.endtime !== "undefined" ||
            typeof localUpdates.days !== "undefined"

          const shiftWillChange =
            (typeof localUpdates.starttime !== "undefined" && localUpdates.starttime !== current.starttime) ||
            (typeof localUpdates.endtime !== "undefined" && localUpdates.endtime !== current.endtime) ||
            (typeof localUpdates.days !== "undefined" && localUpdates.days !== current.days)

          if (shiftFieldsTouched) {
            computedChanges.shiftchange = shiftWillChange ? 1 : 0
          }

          const updateFields = []
          const updateValues = []

          Object.keys(localUpdates).forEach((key) => {
            if (
              [
                "flag1",
                "flag2",
                "flag3",
                "flag4",
                "flag5",
                "flag6",
                "flag7",
                "flag8",
                "active",
                "connected",
                "frequency",
                "userid",
                "depid",
                "starttime",
                "endtime",
                "days",
              ].includes(key)
            ) {
              updateFields.push(`${key} = ?`)
              updateValues.push(localUpdates[key])
            }
          })

          Object.keys(computedChanges).forEach((k) => {
            updateFields.push(`${k} = ?`)
            updateValues.push(computedChanges[k])
          })

          if (updateFields.length > 0) {
            updateFields.push(`setflag = 1`)
            updateValues.push(id, adminId)

            const query = `
            UPDATE userdetails
            SET ${updateFields.join(", ")}, lastupdated = CURRENT_TIMESTAMP
            WHERE id = ? AND loginid = ?
          `

            const [result] = await adminDbPool.execute(query, updateValues)
            if (result.affectedRows > 0) successCount++
          }
        } catch (error) {
          errors.push({ id: update.id, error: error.message })
        }
      }

      res.status(200).json({
        success: true,
        message: "Bulk update completed",
        data: { successCount, totalCount: updates.length, errors },
      })
    } catch (error) {
      console.error("❌ Bulk update user details error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to bulk update user details",
      })
    }
  }

  // 🔍 Get user activity data from userdata{userdetails_id}2 table
  async getUserActivityData(req, res) {
    try {
      const { userdetails_id, date } = req.query
      const adminId = req.user.loginid || req.user.userId

      console.log(`🔍 getUserActivityData called with:`, { userdetails_id, date, adminId })

      if (!userdetails_id || !date) {
        return res.status(400).json({
          success: false,
          message: "userdetails_id and date are required",
        })
      }

      const dateRegex = /^\d{4}-\d{2}-\d{2}$/
      if (!dateRegex.test(date)) {
        return res.status(400).json({
          success: false,
          message: "Date must be in yyyy-mm-dd format",
        })
      }

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)
      const tableName = `userdata${userdetails_id}2`

      const [tableExists] = await adminDbPool.execute(
        `SELECT COUNT(*) as count FROM information_schema.tables 
         WHERE table_schema = DATABASE() AND table_name = ?`,
        [tableName],
      )

      if (tableExists[0].count === 0) {
        return res.json({
          success: true,
          data: [],
          total: 0,
          message: `Activity tracking table ${tableName} does not exist.`,
          tableName: tableName,
          tableExists: false,
        })
      }

      const [activityData] = await adminDbPool.execute(
        `SELECT 
          id,
          loginid,
          userid,
          path,
          time,
          DATE(time) as activity_date,
          TIME(time) as activity_time
         FROM ${tableName} 
         WHERE DATE(time) = ? 
         ORDER BY time ASC`,
        [date],
      )

      res.json({
        success: true,
        data: activityData,
        total: activityData.length,
        tableName: tableName,
        tableExists: true,
        date: date,
      })
    } catch (error) {
      console.error("❌ Get user activity data error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to fetch user activity data",
      })
    }
  }

  // STEP 1: Check if data exists
  async checkDataExists(req, res) {
    try {
      const { loginid, userid, date, userDetailsId } = req.body
      const adminId = req.user.loginid || req.user.userId

      console.log(`🔍 Step 1: Checking if data exists for:`, { loginid, userid, date, userDetailsId, adminId })

      if (!loginid || !userid || !date || !userDetailsId) {
        return res.status(400).json({
          success: false,
          message: "loginid, userid, date, and userDetailsId are required",
        })
      }

      const dateRegex = /^\d{4}-\d{2}-\d{2}$/
      if (!dateRegex.test(date)) {
        return res.status(400).json({
          success: false,
          message: "Date must be in yyyy-mm-dd format",
        })
      }

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)
      const tableName = `userdata${userDetailsId}2`

      const [tableExists] = await adminDbPool.execute(
        `SELECT COUNT(*) as count FROM information_schema.tables 
         WHERE table_schema = DATABASE() AND table_name = ?`,
        [tableName],
      )

      if (tableExists[0].count === 0) {
        return res.json({
          success: false,
          dataExists: false,
          message: `Activity tracking table ${tableName} does not exist.`,
          tableName: tableName,
        })
      }

      const [activityData] = await adminDbPool.execute(
        `SELECT COUNT(*) as count FROM ${tableName} WHERE DATE(time) = ?`,
        [date],
      )

      const recordCount = activityData[0].count

      if (recordCount === 0) {
        return res.json({
          success: false,
          dataExists: false,
          message: `No activity records found in ${tableName} for ${date}.`,
          tableName: tableName,
          recordCount: 0,
        })
      }

      const downloadUrl = `/api/user-details/download-zip?loginid=${loginid}&uid=${userDetailsId}&date=${date}`

      return res.json({
        success: true,
        dataExists: true,
        message: `Found ${recordCount} activity records for ${date}`,
        tableName: tableName,
        recordCount: recordCount,
        downloadUrl: downloadUrl,
      })
    } catch (error) {
      console.error("❌ Check data exists error:", error)
      res.status(500).json({
        success: false,
        dataExists: false,
        message: "Failed to check if data exists",
      })
    }
  }

  // STEP 2: Download ZIP file
  async downloadZipFile(req, res) {
    try {
      const { loginid, uid, date } = req.query

      if (!loginid || !uid || !date) {
        return res.status(400).json({
          success: false,
          message: "loginid, uid, and date are required",
        })
      }

      const dateRegex = /^\d{4}-\d{2}-\d{2}$/
      if (!dateRegex.test(date)) {
        return res.status(400).json({
          success: false,
          message: "Date must be in yyyy-mm-dd format",
        })
      }

      const dotNetApiUrl = `https://api.workdesq.ai/api/Home/GetImage?lid=${loginid}&uid=${uid}&date=${date}`
      console.log(`📡 Calling .NET API: ${dotNetApiUrl}`)

      const response = await fetch(dotNetApiUrl, {
        method: "GET",
        headers: {
          "User-Agent": "Node.js Backend",
          Accept: "application/zip, application/octet-stream, */*",
          Connection: "keep-alive",
        },
        timeout: 120000,
      })

      if (!response.ok) {
        return res.status(response.status).json({
          success: false,
          message: `.NET API Error: ${response.status} ${response.statusText}`,
        })
      }

      const zipBuffer = await response.buffer()
      if (zipBuffer.length === 0) {
        return res.status(404).json({
          success: false,
          message: "No images available for the requested date",
        })
      }

      res.setHeader("Content-Type", "application/zip")
      res.setHeader("Content-Disposition", `attachment; filename="screenshots_${loginid}_${uid}_${date}.zip"`)
      res.setHeader("Content-Length", zipBuffer.length)
      res.setHeader("Access-Control-Expose-Headers", "Content-Disposition")

      res.send(zipBuffer)
    } catch (error) {
      console.error("❌ Download ZIP file error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to download ZIP file",
      })
    }
  }

  // Legacy endpoint for backward compatibility
  async fetchUserImages(req, res) {
    try {
      const { loginid, userid, date } = req.query

      if (!loginid || !userid || !date) {
        return res.status(400).json({
          success: false,
          message: "loginid, userid, and date are required",
        })
      }

      const dateRegex = /^\d{4}-\d{2}-\d{2}$/
      if (!dateRegex.test(date)) {
        return res.status(400).json({
          success: false,
          message: "Date must be in yyyy-mm-dd format",
        })
      }

      const downloadUrl = `/api/user-details/download-zip?loginid=${loginid}&uid=${userid}&date=${date}`
      res.redirect(downloadUrl)
    } catch (error) {
      console.error("❌ Fetch user images error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to fetch user images",
      })
    }
  }

  // 📊 Get user activity data for report generation
  async getUserActivityForReport(req, res) {
    try {
      const { userId, startDate, endDate, days } = req.query
      const adminId = req.user.loginid || req.user.userId

      console.log(`📊 Getting activity for report:`, { userId, startDate, endDate, days, adminId })

      if (!userId || !startDate || !endDate || !days) {
        return res.status(400).json({
          success: false,
          message: "Missing required parameters: userId, startDate, endDate, days",
        })
      }

      const dateRegex = /^\d{4}-\d{2}-\d{2}$/
      if (!dateRegex.test(startDate) || !dateRegex.test(endDate)) {
        return res.status(400).json({
          success: false,
          message: "Dates must be in yyyy-mm-dd format",
        })
      }

      const selectedDays = days.split(",").map((d) => Number.parseInt(d))

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      const [userRows] = await adminDbPool.execute(
        `SELECT 
          ud.id,
          ud.userid,
          ul.email AS user_email,
          ud.devicename,
          ud.depid,
          d.name AS department_name,
          ud.starttime,
          ud.endtime,
          ud.days
         FROM userdetails ud
         LEFT JOIN userlist ul 
           ON ul.id = ud.userid AND ul.loginid = ud.loginid
         LEFT JOIN department d
           ON d.id = ud.depid
         WHERE ud.userid = ? AND ud.loginid = ?`,
        [userId, adminId],
      )

      if (userRows.length === 0) {
        return res.status(404).json({
          success: false,
          message: "User not found",
        })
      }

      const user = userRows[0]
      const userdetailsId = user.id
      const tableName = `userdata${userdetailsId}2`

      const [tableExists] = await adminDbPool.execute(
        `SELECT COUNT(*) as count FROM information_schema.tables 
         WHERE table_schema = DATABASE() AND table_name = ?`,
        [tableName],
      )

      if (tableExists[0].count === 0) {
        console.log(`⚠️ Activity table ${tableName} does not exist`)
        return res.json({
          success: true,
          data: [],
          user: {
            id: user.id,
            userid: user.userid,
            email: user.user_email,
            devicename: user.devicename,
            depid: user.depid,
            department_name: user.department_name,
            starttime: user.starttime,
            endtime: user.endtime,
            days: user.days,
          },
          message: `Activity tracking table ${tableName} does not exist yet.`,
        })
      }

      const start = new Date(startDate)
      const end = new Date(endDate)
      const activityData = []

      for (let d = new Date(start); d <= end; d.setDate(d.getDate() + 1)) {
        const dayOfWeek = d.getDay()
        const currentDate = d.toISOString().split("T")[0]
        const bitValues = [64, 1, 2, 4, 8, 16, 32]
        const dayBit = bitValues[dayOfWeek]

        if (selectedDays.includes(dayOfWeek) && (user.days & dayBit) !== 0) {
          const [startHour, startMin] = (user.starttime || "09:00").split(":").map(Number)
          const [endHour, endMin] = (user.endtime || "17:00").split(":").map(Number)
          const shiftMinutes = endHour * 60 + endMin - (startHour * 60 + startMin)

          const [activityRows] = await adminDbPool.execute(
            `SELECT 
              COUNT(*) as total_records,
              MIN(CONVERT_TZ(time, '+00:00', '+05:30')) as first_activity,
              MAX(CONVERT_TZ(time, '+00:00', '+05:30')) as last_activity
             FROM ${tableName} 
             WHERE DATE(time) = ?`,
            [currentDate],
          )

          if (activityRows.length > 0 && activityRows[0].total_records > 0) {
            const activity = activityRows[0]
            const totalRecords = activity.total_records || 0

            let actualWorkingMinutes = 0
            let normalizedCheckoutTime = null
            const actualCheckoutTime = activity.last_activity
            let overtimeMinutes = 0
            let isStillWorking = false

            if (activity.first_activity && activity.last_activity) {
              const firstActivityDate = new Date(activity.first_activity)
              const lastActivityDate = new Date(activity.last_activity)

              const shiftEndDate = new Date(currentDate + "T00:00:00+05:30")
              shiftEndDate.setHours(endHour, endMin, 0, 0)

              const totalActivityMinutes = Math.floor((lastActivityDate - firstActivityDate) / (1000 * 60))

              const now = new Date()
              const timeSinceLastActivity = Math.floor((now - lastActivityDate) / (1000 * 60))
              isStillWorking = timeSinceLastActivity < 10 && now > shiftEndDate

              if (lastActivityDate > shiftEndDate) {
                normalizedCheckoutTime = shiftEndDate.toISOString()
                overtimeMinutes = Math.floor((lastActivityDate - shiftEndDate) / (1000 * 60))
                actualWorkingMinutes = Math.floor((shiftEndDate - firstActivityDate) / (1000 * 60))
              } else {
                normalizedCheckoutTime = activity.last_activity
                actualWorkingMinutes = totalActivityMinutes
                overtimeMinutes = 0
              }

              actualWorkingMinutes = Math.min(actualWorkingMinutes, shiftMinutes)
            }

            const estimatedWorkingMinutes =
              actualWorkingMinutes > 0 ? actualWorkingMinutes : Math.min(totalRecords * 5, shiftMinutes)
            const idleMinutes = Math.max(0, shiftMinutes - estimatedWorkingMinutes)

            const now = new Date()
            const shiftEndDateForToday = new Date(currentDate + "T00:00:00+05:30")
            shiftEndDateForToday.setHours(endHour, endMin, 0, 0)
            const isShiftOngoing = now < shiftEndDateForToday

            activityData.push({
              date: currentDate,
              dayOfWeek: dayOfWeek,
              workingMinutes: estimatedWorkingMinutes,
              idleMinutes: idleMinutes,
              shiftMinutes: shiftMinutes,
              totalRecords: totalRecords,
              firstActivity: activity.first_activity,
              lastActivity: actualCheckoutTime,
              normalizedCheckout: normalizedCheckoutTime,
              overtimeMinutes: overtimeMinutes,
              shiftStartTime: user.starttime,
              shiftEndTime: user.endtime,
              isShiftOngoing: isShiftOngoing,
              isStillWorking: isStillWorking,
            })
          } else {
            const now = new Date()
            const shiftEndDateForToday = new Date(currentDate + "T00:00:00+05:30")
            shiftEndDateForToday.setHours(endHour, endMin, 0, 0)
            const isShiftOngoing = now < shiftEndDateForToday

            activityData.push({
              date: currentDate,
              dayOfWeek: dayOfWeek,
              workingMinutes: 0,
              idleMinutes: shiftMinutes,
              shiftMinutes: shiftMinutes,
              totalRecords: 0,
              firstActivity: null,
              lastActivity: null,
              normalizedCheckout: null,
              overtimeMinutes: 0,
              shiftStartTime: user.starttime,
              shiftEndTime: user.endtime,
              isShiftOngoing: isShiftOngoing,
              isStillWorking: false,
            })
          }
        }
      }

      return res.status(200).json({
        success: true,
        data: activityData,
        user: {
          id: user.id,
          userid: user.userid,
          email: user.user_email,
          devicename: user.devicename,
          depid: user.depid,
          department_name: user.department_name,
          starttime: user.starttime,
          endtime: user.endtime,
          days: user.days,
        },
        tableName: tableName,
      })
    } catch (error) {
      console.error("❌ Error fetching activity for report:", error)
      return res.status(500).json({
        success: false,
        message: "Failed to fetch activity data for report",
        error: error.message,
      })
    }
  }

  // 📈 Get activity summary for a user across date range
  async getUserActivitySummary(req, res) {
    try {
      const { userId, startDate, endDate } = req.query
      const adminId = req.user.loginid || req.user.userId

      console.log(`📈 Getting activity summary:`, { userId, startDate, endDate, adminId })

      if (!userId || !startDate || !endDate) {
        return res.status(400).json({
          success: false,
          message: "Missing required parameters: userId, startDate, endDate",
        })
      }

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      const [userRows] = await adminDbPool.execute(`SELECT id FROM userdetails WHERE userid = ? AND loginid = ?`, [
        userId,
        adminId,
      ])

      if (userRows.length === 0) {
        return res.status(404).json({
          success: false,
          message: "User not found",
        })
      }

      const userdetailsId = userRows[0].id
      const tableName = `userdata${userdetailsId}2`

      const [tableExists] = await adminDbPool.execute(
        `SELECT COUNT(*) as count FROM information_schema.tables 
         WHERE table_schema = DATABASE() AND table_name = ?`,
        [tableName],
      )

      if (tableExists[0].count === 0) {
        return res.json({
          success: true,
          data: {
            total_days: 0,
            total_records: 0,
            avg_records_per_day: 0,
          },
          message: `Activity tracking table ${tableName} does not exist yet.`,
        })
      }

      const [summaryRows] = await adminDbPool.execute(
        `SELECT 
          COUNT(DISTINCT DATE(time)) as total_days,
          COUNT(*) as total_records,
          AVG(daily_count) as avg_records_per_day
         FROM (
           SELECT DATE(time) as date, COUNT(*) as daily_count
           FROM ${tableName}
           WHERE DATE(time) BETWEEN ? AND ?
           GROUP BY DATE(time)
         ) as daily_stats`,
        [startDate, endDate],
      )

      return res.status(200).json({
        success: true,
        data: summaryRows[0] || {
          total_days: 0,
          total_records: 0,
          avg_records_per_day: 0,
        },
        tableName: tableName,
      })
    } catch (error) {
      console.error("❌ Error fetching activity summary:", error)
      return res.status(500).json({
        success: false,
        message: "Failed to fetch activity summary",
        error: error.message,
      })
    }
  }

  // 🗑️ Uninstall user and free license slot
  async uninstallUser(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      const { userid } = req.params

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      await adminDbPool.execute(
        `UPDATE userdetails
         SET 
           connected = 2,
           active = 0,
           flag1 = 0,
           flag2 = 0,
           flag3 = 0,
           flag4 = 0,
           flag5 = 0,
           flag6 = 0,
           flag7 = 0,
           flag8 = 0,
           setflag = 1,
           lastupdated = CURRENT_TIMESTAMP
         WHERE userid = ? AND loginid = ?`,
        [userid, adminId],
      )

      await adminDbPool.execute(
        `DELETE FROM userlist
         WHERE id = ? AND loginid = ?`,
        [userid, adminId],
      )

      res.json({
        success: true,
        message: "Employee uninstalled successfully",
      })
    } catch (error) {
      console.error("❌ Uninstall error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to uninstall employee",
      })
    }
  }
}

module.exports = new UserDetailsController()
