function calculateShiftBoundMinutes(activityStart, activityEnd, shiftStart, shiftEnd) {
  if (!activityStart || !activityEnd || !shiftStart || !shiftEnd) return 0

  const actStart = new Date(activityStart)
  const actEnd = new Date(activityEnd)

  const shiftStartTime = new Date(activityStart)
  const shiftEndTime = new Date(activityStart)

  const [sh, sm] = shiftStart.split(":").map(Number)
  const [eh, em] = shiftEnd.split(":").map(Number)

  shiftStartTime.setHours(sh, sm, 0, 0)
  shiftEndTime.setHours(eh, em, 0, 0)

  // 🌙 Night shift support
  if (shiftEndTime <= shiftStartTime) {
    shiftEndTime.setDate(shiftEndTime.getDate() + 1)
  }

  const effectiveStart = new Date(Math.max(actStart, shiftStartTime))
  const effectiveEnd = new Date(Math.min(actEnd, shiftEndTime))

  const diffMinutes = (effectiveEnd - effectiveStart) / (1000 * 60)
  return diffMinutes > 0 ? diffMinutes : 0
}

class UserActivityController {
  constructor() {
    this.getUserActivityData = this.getUserActivityData.bind(this)
    this.getUserActivitySummary = this.getUserActivitySummary.bind(this)
  }



  
  // Get user activity data from userdata{userid}{serviceid} table
  async getUserActivityData(req, res) {
    try {
      const { userId, serviceId } = req.params
      const { startDate, endDate, limit = 100, useShiftFilter } = req.query
      const adminId = req.user.userId

      console.log(`🔍 Fetching activity data for user ${userId}, service ${serviceId} by admin ${adminId}`)

      const databaseConfig = require("../config/database")
      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      const dbExists = await databaseConfig.testAdminDatabase(adminId)
      if (!dbExists) {
        return res.status(400).json({
          success: false,
          message: `Admin database edb${adminId} is not accessible`,
          database: `edb${adminId}`,
          accessible: false,
        })
      }

      const tableName = `userdata${userId}${serviceId}`
      console.log(`📊 Querying table: ${tableName}`)

      const [tableExists] = await adminDbPool.execute(
        "SELECT COUNT(*) as count FROM information_schema.tables WHERE table_schema = ? AND table_name = ?",
        [`edb${adminId}`, tableName],
      )

      if (tableExists[0].count === 0) {
        return res.json({
          success: true,
          data: [],
          message: `No activity data found for user ${userId} service ${serviceId}. Table ${tableName} does not exist.`,
          tableName: tableName,
          database: `edb${adminId}`,
        })
      }

      let userShiftInfo = null
      if (useShiftFilter) {
        const [userDetails] = await adminDbPool.execute(
          "SELECT starttime, endtime, days FROM userdetails WHERE id = ? AND loginid = ?",
          [userId, adminId],
        )

        if (userDetails.length > 0) {
          userShiftInfo = userDetails[0]
        }
      }

      const [columns] = await adminDbPool.execute(
        "SELECT COLUMN_NAME FROM information_schema.COLUMNS WHERE table_schema = ? AND table_name = ?",
        [`edb${adminId}`, tableName],
      )

      console.log(
        `📋 Available columns in ${tableName}:`,
        columns.map((col) => col.COLUMN_NAME),
      )

      let selectColumns = []
      const availableColumns = columns.map((col) => col.COLUMN_NAME.toLowerCase())

      if (availableColumns.includes("id")) selectColumns.push("id")
      if (availableColumns.includes("loginid")) selectColumns.push("loginid")
      if (availableColumns.includes("userid")) selectColumns.push("userid")

      if (availableColumns.includes("duration")) {
        selectColumns.push("duration")
      } else if (availableColumns.includes("time_duration")) {
        selectColumns.push("time_duration as duration")
      } else if (availableColumns.includes("session_duration")) {
        selectColumns.push("session_duration as duration")
      } else {
        if (availableColumns.includes("startdatetime") && availableColumns.includes("enddatetime")) {
          selectColumns.push("TIMESTAMPDIFF(MINUTE, startdatetime, enddatetime) as duration")
        } else {
          selectColumns.push("0 as duration")
        }
      }

      if (availableColumns.includes("status")) {
        selectColumns.push("status")
      } else if (availableColumns.includes("activity_status")) {
        selectColumns.push("activity_status as status")
      } else if (availableColumns.includes("state")) {
        selectColumns.push("state as status")
      } else if (availableColumns.includes("activity_type")) {
        selectColumns.push("activity_type as status")
      } else {
        selectColumns.push('"Working" as status')
      }

      if (availableColumns.includes("startdatetime")) selectColumns.push("startdatetime")
      if (availableColumns.includes("enddatetime")) selectColumns.push("enddatetime")
      if (availableColumns.includes("timecreated")) selectColumns.push("timecreated")
      if (availableColumns.includes("created_at")) selectColumns.push("created_at as timecreated")
      if (availableColumns.includes("timestamp")) selectColumns.push("timestamp as timecreated")

      // Add appname (and common variants) to the select list if present
      if (availableColumns.includes("appname")) {
        selectColumns.push("appname")
      } else if (availableColumns.includes("app_name")) {
        selectColumns.push("app_name as appname")
      } else if (availableColumns.includes("application")) {
        selectColumns.push("application as appname")
      }

      if (selectColumns.length === 0) {
        selectColumns = ["*"]
      }

      let query = `SELECT ${selectColumns.join(", ")} FROM ${tableName} WHERE 1=1`
      const queryParams = []

      if (availableColumns.includes("loginid")) {
        query += " AND loginid = ?"
        queryParams.push(adminId)
      }

      if (availableColumns.includes("startdatetime")) {
        if (startDate) {
          query += " AND DATE(startdatetime) >= ?"
          queryParams.push(startDate)
        }
        if (endDate) {
          query += " AND DATE(startdatetime) <= ?"
          queryParams.push(endDate)
        }

        if (userShiftInfo && userShiftInfo.starttime && userShiftInfo.endtime) {
          const startTime = userShiftInfo.starttime
          const endTime = userShiftInfo.endtime

          // Handle night shifts (e.g., 22:00 to 06:00)
          if (startTime > endTime) {
            query += " AND (TIME(startdatetime) >= ? OR TIME(startdatetime) <= ?)"
            queryParams.push(startTime, endTime)
          } else {
            // Normal day shift
            query += " AND TIME(startdatetime) >= ? AND TIME(startdatetime) <= ?"
            queryParams.push(startTime, endTime)
          }
        }

        query += " ORDER BY startdatetime DESC"
      } else if (availableColumns.includes("timecreated")) {
        if (startDate) {
          query += " AND DATE(timecreated) >= ?"
          queryParams.push(startDate)
        }
        if (endDate) {
          query += " AND DATE(timecreated) <= ?"
          queryParams.push(endDate)
        }
        query += " ORDER BY timecreated DESC"
      } else if (availableColumns.includes("created_at")) {
        if (startDate) {
          query += " AND DATE(created_at) >= ?"
          queryParams.push(startDate)
        }
        if (endDate) {
          query += " AND DATE(created_at) <= ?"
          queryParams.push(endDate)
        }
        query += " ORDER BY created_at DESC"
      }

      console.log(`🔍 Executing query: ${query}`)
      console.log(`📝 Query params:`, queryParams)

      const [activityData] = await adminDbPool.execute(query, queryParams)

      let filteredData = activityData
      if (userShiftInfo && userShiftInfo.days) {
  filteredData = activityData.filter((row) => {
    try {
      const activityDate = new Date(row.startdatetime || row.timecreated || row.created_at)
      const dayOfWeek = activityDate.getDay() // 0=Sun, 1=Mon, ..., 6=Sat

      // New mapping: Mon=1, Tue=2, Wed=4, Thu=8, Fri=16, Sat=32, Sun=64
      const bitValues = [64, 1, 2, 4, 8, 16, 32]
      const dayBit = bitValues[dayOfWeek]

      return (userShiftInfo.days & dayBit) !== 0
    } catch {
      return true
    }
  })
}

const processedData = filteredData.map((row) => ({
  ...row, // ✅ Spread first so custom mappings don’t get overwritten
  id: row.id || 0,
  loginid: row.loginid || adminId,
  userid: row.userid || userId,
duration: userShiftInfo
  ? calculateShiftBoundMinutes(
      row.startdatetime,
      row.enddatetime,
      userShiftInfo.starttime,
      userShiftInfo.endtime
    )
  : Number.parseFloat(row.duration) || 0,

  status: row.status || "Working",
  startdatetime: row.startdatetime || row.timecreated || row.created_at || new Date().toISOString(),
  enddatetime: row.enddatetime || null,
  timecreated: row.timecreated || row.created_at || row.startdatetime || new Date().toISOString(),
  appname: row.appname || row.app_name || row.application || null, // ✅ ensure appname is preserved
}))


      console.log(`✅ Found ${processedData.length} activity records in table: ${tableName}`)

      res.json({
        success: true,
        data: processedData,
        total: processedData.length,
        tableName: tableName,
        database: `edb${adminId}`,
        userId: Number.parseInt(userId),
        serviceId: Number.parseInt(serviceId),
        availableColumns: availableColumns,
        shiftFiltered: !!userShiftInfo,
        shiftInfo: userShiftInfo,
      })
    } catch (error) {
      console.error("❌ Get user activity data error:", error)

      let errorMessage = "Failed to fetch user activity data"
      if (error.code === "ER_DBACCESS_DENIED_ERROR") {
        errorMessage = `Database access denied for edb${req.user.userId}`
      } else if (error.code === "ER_BAD_DB_ERROR") {
        errorMessage = `Database edb${req.user.userId} does not exist`
      } else if (error.code === "ER_NO_SUCH_TABLE") {
        errorMessage = `Activity table does not exist for this user and service`
      } else if (error.code === "ER_BAD_FIELD_ERROR") {
        errorMessage = `Database schema mismatch. Please check table structure.`
      }

      res.status(500).json({
        success: false,
        message: errorMessage,
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
        sqlError: process.env.NODE_ENV === "development" ? error.sql : undefined,
      })
    }
  }

  // Get user activity summary (total working time, idle time, etc.)
  async getUserActivitySummary(req, res) {
    try {
      const { userId, serviceId } = req.params
      const { startDate, endDate } = req.query
      const adminId = req.user.userId

      console.log(`📈 Fetching activity summary for user ${userId}, service ${serviceId}`)

      const databaseConfig = require("../config/database")
      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      const tableName = `userdata${userId}${serviceId}`

      // ✅ Check if table exists
      const [tableExists] = await adminDbPool.execute(
        "SELECT COUNT(*) as count FROM information_schema.tables WHERE table_schema = ? AND table_name = ?",
        [`edb${adminId}`, tableName],
      )

      if (tableExists[0].count === 0) {
        return res.json({
          success: true,
          data: {
            totalWorkingTime: 0,
            totalIdleTime: 0,
            totalSessions: 0,
            averageSessionDuration: 0,
            workingPercentage: 0,
            idlePercentage: 0,
          },
          message: `No activity data found for user ${userId} service ${serviceId}`,
        })
      }

      // ✅ Detect available columns
      const [columns] = await adminDbPool.execute(
        "SELECT COLUMN_NAME FROM information_schema.COLUMNS WHERE table_schema = ? AND table_name = ?",
        [`edb${adminId}`, tableName],
      )

      const availableColumns = columns.map((col) => col.COLUMN_NAME.toLowerCase())
      let durationColumn = "duration"
      let statusColumn = "status"

      // Handle duration variations
      if (availableColumns.includes("time_duration")) durationColumn = "time_duration"
      else if (availableColumns.includes("session_duration")) durationColumn = "session_duration"
      else if (availableColumns.includes("startdatetime") && availableColumns.includes("enddatetime"))
        durationColumn = "TIMESTAMPDIFF(MINUTE, startdatetime, enddatetime)"
      else durationColumn = "0"

      // Handle status variations
      if (availableColumns.includes("activity_status")) statusColumn = "activity_status"
      else if (availableColumns.includes("state")) statusColumn = "state"
      else if (availableColumns.includes("activity_type")) statusColumn = "activity_type"
      else if (!availableColumns.includes("status")) statusColumn = '"Unknown"'

      let query = `
        SELECT 
          ${statusColumn} as status,
          SUM(${durationColumn}) as total_duration,
          COUNT(*) as session_count
        FROM ${tableName}
        WHERE 1=1
      `
      const queryParams = []

      // Filter by admin loginid if available
      if (availableColumns.includes("loginid")) {
        query += " AND loginid = ?"
        queryParams.push(adminId)
      }

      // ✅ Date filtering
      if (availableColumns.includes("startdatetime")) {
        if (startDate) {
          query += " AND DATE(startdatetime) >= ?"
          queryParams.push(startDate)
        }
        if (endDate) {
          query += " AND DATE(startdatetime) <= ?"
          queryParams.push(endDate)
        }
      } else if (availableColumns.includes("timecreated")) {
        if (startDate) {
          query += " AND DATE(timecreated) >= ?"
          queryParams.push(startDate)
        }
        if (endDate) {
          query += " AND DATE(timecreated) <= ?"
          queryParams.push(endDate)
        }
      }

      query += ` GROUP BY ${statusColumn}`
      console.log(`🔍 Summary Query: ${query}`, queryParams)

      const [summaryData] = await adminDbPool.execute(query, queryParams)

      // ✅ If no records in the date range
      if (!summaryData || summaryData.length === 0) {
        return res.json({
          success: true,
          data: {
            totalWorkingTime: 0,
            totalIdleTime: 0,
            totalSessions: 0,
            averageSessionDuration: 0,
            workingPercentage: 0,
            idlePercentage: 0,
          },
          message: `No activity data found in range ${startDate} - ${endDate}`,
        })
      }

      // ✅ Process records
      let totalWorkingTime = 0
      let totalIdleTime = 0
      let totalSessions = 0

      summaryData.forEach((row) => {
        const duration = Number.parseFloat(row.total_duration) || 0
        const sessions = Number.parseInt(row.session_count) || 0
        const status = (row.status || "").toLowerCase()

        if (status.includes("work") || status.includes("active")) {
          totalWorkingTime += duration
        } else if (status.includes("idle") || status.includes("inactive")) {
          totalIdleTime += duration
        }
        totalSessions += sessions
      })

      const totalTime = totalWorkingTime + totalIdleTime
      const workingPercentage = totalTime > 0 ? (totalWorkingTime / totalTime) * 100 : 0
      const idlePercentage = totalTime > 0 ? (totalIdleTime / totalTime) * 100 : 0
      const averageSessionDuration = totalSessions > 0 ? totalTime / totalSessions : 0

      res.json({
        success: true,
        data: {
          totalWorkingTime: Math.round(totalWorkingTime * 100) / 100,
          totalIdleTime: Math.round(totalIdleTime * 100) / 100,
          totalSessions,
          averageSessionDuration: Math.round(averageSessionDuration * 100) / 100,
          workingPercentage: Math.round(workingPercentage * 100) / 100,
          idlePercentage: Math.round(idlePercentage * 100) / 100,
        },
        tableName,
        userId: Number.parseInt(userId),
        serviceId: Number.parseInt(serviceId),
      })
    } catch (error) {
      console.error("❌ Get user activity summary error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to fetch user activity summary",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }
}

module.exports = new UserActivityController()
