const databaseConfig = require("../config/database")

class AppController {
  constructor() {
    this.getAllApps = this.getAllApps.bind(this)
    this.addApp = this.addApp.bind(this)
    this.updateApp = this.updateApp.bind(this)
    this.deleteApp = this.deleteApp.bind(this)
    this.getUserExcludedApps = this.getUserExcludedApps.bind(this)
    this.updateUserExcludedApps = this.updateUserExcludedApps.bind(this)
    this.updateAllUsersExcludedApps = this.updateAllUsersExcludedApps.bind(this)
    this.addExeToUserExcluded = this.addExeToUserExcluded.bind(this)
    this.addExeToAllUsersExcluded = this.addExeToAllUsersExcluded.bind(this)
    this.checkRunningApps = this.checkRunningApps.bind(this)
  }

  // Get all applications from the master app table (in emp_clouddb)
  async getAllApps(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      console.log(`Fetching all applications for admin ID: ${adminId}`)

      // Use main pool for emp_clouddb.app table (master app list)
      const mainPool = databaseConfig.empCloudDbPool

      const [apps] = await mainPool.execute(`
        SELECT 
          id,
          name,
          exename,
          ApplicationTitle,
          UserFriendlyName
        FROM emp_clouddb.app
        ORDER BY name ASC
      `)

      res.status(200).json({
        success: true,
        message: "Applications retrieved successfully",
        data: apps,
        total: apps.length,
        database: "emp_clouddb",
      })
    } catch (error) {
      console.error("Get all apps error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to retrieve applications",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Add a new application to the master app table (in emp_clouddb)
  async addApp(req, res) {
    try {
      const { name, exename, ApplicationTitle, UserFriendlyName } = req.body

      console.log(`Adding new application: ${name}`)

      if (!name || !exename) {
        return res.status(400).json({
          success: false,
          message: "Application name and exe name are required",
        })
      }

      // Use main pool for emp_clouddb.app table
      const mainPool = databaseConfig.empCloudDbPool

      // Check if app with same exename already exists
      const [existing] = await mainPool.execute(
        "SELECT id FROM emp_clouddb.app WHERE exename = ?",
        [exename]
      )

      if (existing.length > 0) {
        return res.status(400).json({
          success: false,
          message: "Application with this exe name already exists",
        })
      }

      const [result] = await mainPool.execute(
        `INSERT INTO emp_clouddb.app (name, exename, ApplicationTitle, UserFriendlyName)
         VALUES (?, ?, ?, ?)`,
        [name, exename, ApplicationTitle || null, UserFriendlyName || name]
      )

      res.status(201).json({
        success: true,
        message: "Application added successfully",
        data: {
          id: result.insertId,
          name,
          exename,
          ApplicationTitle,
          UserFriendlyName: UserFriendlyName || name,
        },
      })
    } catch (error) {
      console.error("Add app error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to add application",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Update an existing application (in emp_clouddb)
  async updateApp(req, res) {
    try {
      const { id } = req.params
      const { name, exename, ApplicationTitle, UserFriendlyName } = req.body

      console.log(`Updating application ID: ${id}`)

      // Use main pool for emp_clouddb.app table
      const mainPool = databaseConfig.empCloudDbPool

      // Check if app exists
      const [existing] = await mainPool.execute(
        "SELECT id FROM emp_clouddb.app WHERE id = ?",
        [id]
      )

      if (existing.length === 0) {
        return res.status(404).json({
          success: false,
          message: "Application not found",
        })
      }

      // Check if new exename conflicts with another app
      if (exename) {
        const [conflict] = await mainPool.execute(
          "SELECT id FROM emp_clouddb.app WHERE exename = ? AND id != ?",
          [exename, id]
        )

        if (conflict.length > 0) {
          return res.status(400).json({
            success: false,
            message: "Another application with this exe name already exists",
          })
        }
      }

      const updateFields = []
      const updateValues = []

      if (name) {
        updateFields.push("name = ?")
        updateValues.push(name)
      }
      if (exename) {
        updateFields.push("exename = ?")
        updateValues.push(exename)
      }
      if (ApplicationTitle !== undefined) {
        updateFields.push("ApplicationTitle = ?")
        updateValues.push(ApplicationTitle)
      }
      if (UserFriendlyName !== undefined) {
        updateFields.push("UserFriendlyName = ?")
        updateValues.push(UserFriendlyName)
      }

      if (updateFields.length === 0) {
        return res.status(400).json({
          success: false,
          message: "No valid fields to update",
        })
      }

      updateValues.push(id)

      await mainPool.execute(
        `UPDATE emp_clouddb.app SET ${updateFields.join(", ")} WHERE id = ?`,
        updateValues
      )

      res.status(200).json({
        success: true,
        message: "Application updated successfully",
      })
    } catch (error) {
      console.error("Update app error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to update application",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Delete an application (from emp_clouddb)
  async deleteApp(req, res) {
    try {
      const { id } = req.params

      console.log(`Deleting application ID: ${id}`)

      // Use main pool for emp_clouddb.app table
      const mainPool = databaseConfig.empCloudDbPool

      const [result] = await mainPool.execute(
        "DELETE FROM emp_clouddb.app WHERE id = ?",
        [id]
      )

      if (result.affectedRows === 0) {
        return res.status(404).json({
          success: false,
          message: "Application not found",
        })
      }

      res.status(200).json({
        success: true,
        message: "Application deleted successfully",
      })
    } catch (error) {
      console.error("Delete app error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to delete application",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Get user's excluded apps configuration (from userdetails in user's database)
  async getUserExcludedApps(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      const { userid } = req.params

      console.log(`Fetching excluded apps for user ID: ${userid}`)

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)
      const dbExists = await databaseConfig.testAdminDatabase(adminId)

      if (!dbExists) {
        return res.status(400).json({
          success: false,
          message: `Admin database edb${adminId} is not accessible.`,
        })
      }

      // Get user details with excluded apps (excludeapp column)
      const [userDetails] = await adminDbPool.execute(
        `SELECT id, userid, loginid, excludeapp, exappchange
         FROM userdetails
         WHERE userid = ? AND loginid = ?`,
        [userid, adminId]
      )

      if (userDetails.length === 0) {
        return res.status(404).json({
          success: false,
          message: "User not found",
        })
      }

      const user = userDetails[0]
      // excludeapp is varchar(500), stores comma-separated exenames
      const excludedApps = user.excludeapp
        ? user.excludeapp.split(",").filter((app) => app.trim() !== "")
        : []

      res.status(200).json({
        success: true,
        message: "User excluded apps retrieved successfully",
        data: {
          userDetailsId: user.id,
          userid: user.userid,
          excludedApps: excludedApps,
          exappchange: user.exappchange || 0,
          hasCustomRules: user.exappchange === 1,
        },
      })
    } catch (error) {
      console.error("Get user excluded apps error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to retrieve user excluded apps",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Update user's excluded apps (save blocked applications to userdetails)
  async updateUserExcludedApps(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      const { userid } = req.params
      const { excludedApps } = req.body

      console.log(`Updating excluded apps for user ID: ${userid}`)

      if (!Array.isArray(excludedApps)) {
        return res.status(400).json({
          success: false,
          message: "excludedApps must be an array",
        })
      }

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)
      const dbExists = await databaseConfig.testAdminDatabase(adminId)

      if (!dbExists) {
        return res.status(400).json({
          success: false,
          message: `Admin database edb${adminId} is not accessible.`,
        })
      }

      // Convert array to comma-separated string (max 500 chars for excludeapp column)
      const excludedAppsString = excludedApps.join(",")

      // Update userdetails table (excludeapp column)
      const [result] = await adminDbPool.execute(
        `UPDATE userdetails 
         SET excludeapp = ?, exappchange = 1, lastupdated = NOW()
         WHERE userid = ? AND loginid = ?`,
        [excludedAppsString, userid, adminId]
      )

      if (result.affectedRows === 0) {
        return res.status(404).json({
          success: false,
          message: "User not found",
        })
      }

      res.status(200).json({
        success: true,
        message: "User excluded apps updated successfully",
        data: {
          userid,
          excludedApps,
          excludedAppsCount: excludedApps.length,
          exappchange: 1,
        },
      })
    } catch (error) {
      console.error("Update user excluded apps error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to update user excluded apps",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Update ALL users' excluded apps at once
  async updateAllUsersExcludedApps(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      const { excludedApps } = req.body

      console.log(`Updating excluded apps for ALL users under admin ID: ${adminId}`)

      if (!Array.isArray(excludedApps)) {
        return res.status(400).json({
          success: false,
          message: "excludedApps must be an array",
        })
      }

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)
      const dbExists = await databaseConfig.testAdminDatabase(adminId)

      if (!dbExists) {
        return res.status(400).json({
          success: false,
          message: `Admin database edb${adminId} is not accessible.`,
        })
      }

      // Convert array to comma-separated string
      const excludedAppsString = excludedApps.join(",")

      // Update ALL userdetails records for this admin
      const [result] = await adminDbPool.execute(
        `UPDATE userdetails 
         SET excludeapp = ?, exappchange = 1, lastupdated = NOW()
         WHERE loginid = ?`,
        [excludedAppsString, adminId]
      )

      res.status(200).json({
        success: true,
        message: "All users' excluded apps updated successfully",
        data: {
          affectedUsers: result.affectedRows,
          excludedApps,
          excludedAppsCount: excludedApps.length,
        },
      })
    } catch (error) {
      console.error("Update all users excluded apps error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to update all users excluded apps",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Add exe to a specific user's excluded list (simple add - just exename)
  async addExeToUserExcluded(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      const { userid } = req.params
      const { exename } = req.body

      console.log(`Adding exe ${exename} to excluded list for user ID: ${userid}`)

      if (!exename) {
        return res.status(400).json({
          success: false,
          message: "exename is required",
        })
      }

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)
      const dbExists = await databaseConfig.testAdminDatabase(adminId)

      if (!dbExists) {
        return res.status(400).json({
          success: false,
          message: `Admin database edb${adminId} is not accessible.`,
        })
      }

      // Get current excluded apps for user
      const [userDetails] = await adminDbPool.execute(
        `SELECT excludeapp FROM userdetails WHERE userid = ? AND loginid = ?`,
        [userid, adminId]
      )

      if (userDetails.length === 0) {
        return res.status(404).json({
          success: false,
          message: "User not found",
        })
      }

      // Add exename to excluded apps if not already present
      let excludedApps = userDetails[0].excludeapp
        ? userDetails[0].excludeapp.split(",").filter((app) => app.trim() !== "")
        : []

      if (!excludedApps.includes(exename)) {
        excludedApps.push(exename)
      }

      // Update userdetails
      await adminDbPool.execute(
        `UPDATE userdetails 
         SET excludeapp = ?, exappchange = 1, lastupdated = NOW()
         WHERE userid = ? AND loginid = ?`,
        [excludedApps.join(","), userid, adminId]
      )

      res.status(200).json({
        success: true,
        message: "Application added to user's excluded list",
        data: {
          userid,
          exename,
          excludedApps,
        },
      })
    } catch (error) {
      console.error("Add exe to user excluded error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to add application to user's excluded list",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Add exe to ALL users' excluded list
  async addExeToAllUsersExcluded(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      const { exename } = req.body

      console.log(`Adding exe ${exename} to excluded list for ALL users under admin ID: ${adminId}`)

      if (!exename) {
        return res.status(400).json({
          success: false,
          message: "exename is required",
        })
      }

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)
      const dbExists = await databaseConfig.testAdminDatabase(adminId)

      if (!dbExists) {
        return res.status(400).json({
          success: false,
          message: `Admin database edb${adminId} is not accessible.`,
        })
      }

      // Get all users for this admin
      const [users] = await adminDbPool.execute(
        `SELECT userid, excludeapp FROM userdetails WHERE loginid = ?`,
        [adminId]
      )

      let updatedCount = 0

      for (const user of users) {
        let excludedApps = user.excludeapp
          ? user.excludeapp.split(",").filter((app) => app.trim() !== "")
          : []

        if (!excludedApps.includes(exename)) {
          excludedApps.push(exename)

          await adminDbPool.execute(
            `UPDATE userdetails 
             SET excludeapp = ?, exappchange = 1, lastupdated = NOW()
             WHERE userid = ? AND loginid = ?`,
            [excludedApps.join(","), user.userid, adminId]
          )
          updatedCount++
        }
      }

      res.status(200).json({
        success: true,
        message: "Application added to all users' excluded list",
        data: {
          exename,
          affectedUsers: updatedCount,
          totalUsers: users.length,
        },
      })
    } catch (error) {
      console.error("Add exe to all users excluded error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to add application to all users' excluded list",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Check running apps against excluded list (called by desktop agent)
  async checkRunningApps(req, res) {
    try {
      const adminId = req.user.loginid || req.user.userId
      const { userid, runningApps } = req.body

      console.log(`Checking running apps for user ID: ${userid}`)

      if (!userid || !Array.isArray(runningApps)) {
        return res.status(400).json({
          success: false,
          message: "userid and runningApps array are required",
        })
      }

      const adminDbPool = databaseConfig.getAdminDbPool(adminId)
      const mainPool = databaseConfig.empCloudDbPool

      // Get user's excluded apps (excludeapp column from userdetails)
      const [userDetails] = await adminDbPool.execute(
        `SELECT excludeapp FROM userdetails WHERE userid = ? AND loginid = ?`,
        [userid, adminId]
      )

      if (userDetails.length === 0) {
        return res.status(404).json({
          success: false,
          message: "User not found",
        })
      }

      const excludedAppsString = userDetails[0].excludeapp || ""
      const excludedApps = excludedAppsString
        .split(",")
        .map((app) => app.trim().toLowerCase())
        .filter((app) => app !== "")

      // Find blocked apps that are currently running
      const blockedRunningApps = runningApps.filter((app) =>
        excludedApps.includes(app.toLowerCase())
      )

      // Check for unknown apps (not in master list in emp_clouddb)
      const [allApps] = await mainPool.execute("SELECT exename FROM emp_clouddb.app")
      const knownApps = allApps.map((app) => app.exename.toLowerCase())

      const unknownApps = runningApps.filter(
        (app) => !knownApps.includes(app.toLowerCase())
      )

      res.status(200).json({
        success: true,
        data: {
          blockedApps: blockedRunningApps,
          blockedCount: blockedRunningApps.length,
          unknownApps: unknownApps,
          unknownCount: unknownApps.length,
          shouldBlock: blockedRunningApps.length > 0,
          shouldPromptForUnknown: unknownApps.length > 0,
        },
      })
    } catch (error) {
      console.error("Check running apps error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to check running apps",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }
}

module.exports = new AppController()
