const bcrypt = require("bcryptjs")
const jwt = require("jsonwebtoken")
const fs = require("fs")
const path = require("path")

// Try to require dependencies
let nodemailer, csv, multer, fetch
try {
  nodemailer = require("nodemailer")
} catch (e) {
  console.warn("⚠️ nodemailer not installed - email functionality disabled")
}

try {
  csv = require("csv-parser")
} catch (e) {
  console.warn("⚠️ csv-parser not installed - CSV upload functionality disabled")
}

try {
  multer = require("multer")
} catch (e) {
  console.warn("⚠️ multer not installed - file upload functionality disabled")
}

try {
  fetch = require("node-fetch")
} catch (e) {
  console.warn("⚠️ node-fetch not installed - API calls may not work")
}

class EmployeeController {
  constructor() {
    this.addEmployee = this.addEmployee.bind(this)
    this.getEmployees = this.getEmployees.bind(this)
    this.uploadCSV = this.uploadCSV.bind(this)
    this.sendEmail = this.sendEmail.bind(this)
    this.resendEmail = this.resendEmail.bind(this)
    this.bulkEmail = this.bulkEmail.bind(this)
    this.updateEmployee = this.updateEmployee.bind(this)
    this.deleteEmployee = this.deleteEmployee.bind(this)
    this.checkAdminDatabase = this.checkAdminDatabase.bind(this)
  }

  // Check admin database status
  async checkAdminDatabase(req, res) {
    try {
      const adminId = req.user.userId
      const databaseConfig = require("../config/database")

      console.log(`🔍 Checking database access for admin ID: ${adminId}`)

      // Test if admin database exists and is accessible
      const dbExists = await databaseConfig.testAdminDatabase(adminId)

      if (dbExists) {
        // Try to query the userlist table
        const adminDbPool = databaseConfig.getAdminDbPool(adminId)
        const [result] = await adminDbPool.execute("SELECT COUNT(*) as count FROM userlist")

        res.json({
          success: true,
          message: `Admin database edb${adminId} is accessible`,
          data: {
            database: `edb${adminId}`,
            accessible: true,
            employeeCount: result[0].count,
          },
        })
      } else {
        res.status(400).json({
          success: false,
          message: `Admin database edb${adminId} is not accessible`,
          data: {
            database: `edb${adminId}`,
            accessible: false,
            suggestion:
              "Please contact your database administrator to grant permissions or ensure the .NET API has been called to initialize your organization.",
          },
        })
      }
    } catch (error) {
      console.error("❌ Check admin database error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to check admin database",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Get all employees for the current admin
async getEmployees(req, res) {
  try {
    const adminId = req.user.userId
    const databaseConfig = require("../config/database")

    console.log(`🔍 Fetching employees for admin ID: ${adminId} from database: edb${adminId}`)

    // Get admin-specific database pool
    const adminDbPool = databaseConfig.getAdminDbPool(adminId)

    // Test if admin database exists and is accessible
    const dbExists = await databaseConfig.testAdminDatabase(adminId)
    if (!dbExists) {
      return res.json({
        success: true,
        data: [],
        total: 0,
        message: `Admin database edb${adminId} is not accessible. This could be due to:
1. Database permissions not granted
2. Database not yet created
3. Network connectivity issues`,
        database: `edb${adminId}`,
        accessible: false,
      })
    }

    /**
     * 🔥 MAIN CHANGE IS HERE
     * LEFT JOIN userdetails to know
     * whether employee has installed software or not
     */
    const [employees] = await adminDbPool.execute(
      `
      SELECT 
        u.id,
        u.loginid,
        u.email,
        u.link,
        u.sendemail,
        u.datecreated,
        u.lastupdated,
        CASE 
          WHEN ud.id IS NULL THEN false
          ELSE true
        END AS hasUserDetails
      FROM userlist u
      LEFT JOIN userdetails ud 
        ON ud.userid = u.id AND ud.loginid = u.loginid
        WHERE (ud.connected IS NULL OR ud.connected != 2)
      ORDER BY u.datecreated DESC
      `
    )

    console.log(`✅ Found ${employees.length} employees in database: edb${adminId}`)

    res.json({
      success: true,
      data: employees,
      total: employees.length,
      database: `edb${adminId}`,
      accessible: true,
    })
  } catch (error) {
    console.error("❌ Get employees error:", error)

    let errorMessage = "Failed to fetch employees"
    if (error.code === "ER_DBACCESS_DENIED_ERROR") {
      errorMessage = `Database access denied for edb${req.user.userId}. Please contact your database administrator.`
    } else if (error.code === "ER_BAD_DB_ERROR") {
      errorMessage = `Database edb${req.user.userId} does not exist. Please ensure your organization has been initialized.`
    }

    res.status(500).json({
      success: false,
      message: errorMessage,
      database: `edb${req.user.userId}`,
      accessible: false,
      error: process.env.NODE_ENV === "development" ? error.message : undefined,
    })
  }
}

  // Add single employee manually
 async addEmployee(req, res) {
  const { email, sendEmailFlag = false } = req.body
  const adminId = req.user.userId

  try {
    /* ================= VALIDATION ================= */
    if (!email) {
      return res.status(400).json({
        success: false,
        message: "Email is required",
      })
    }

    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
    if (!emailRegex.test(email)) {
      return res.status(400).json({
        success: false,
        message: "Invalid email format",
      })
    }

    const databaseConfig = require("../config/database")
    const empCloudDbPool = req.app.locals.empCloudDbPool

    console.log(`🔍 Adding employee to database: edb${adminId}`)

    /* ================= ADMIN DB CHECK ================= */
    const adminDbPool = databaseConfig.getAdminDbPool(adminId)

    const dbExists = await databaseConfig.testAdminDatabase(adminId)
    if (!dbExists) {
      return res.status(400).json({
        success: false,
        message: `Admin database edb${adminId} is not accessible.`,
        database: `edb${adminId}`,
        accessible: false,
      })
    }

    /* ================= LICENSE CHECK (MAIN LOGIC) ================= */

    // 1. Fetch license
    const [[license]] = await empCloudDbPool.execute(
      "SELECT nouser FROM license WHERE loginid = ?",
      [adminId]
    )

    if (!license) {
      return res.status(400).json({
        success: false,
        message: "License not found",
      })
    }

    // 2. Count existing employees
    const [countResult] = await adminDbPool.execute(
      "SELECT COUNT(*) AS total FROM userlist WHERE loginid = ?",
      [adminId]
    )

    const totalEmployees = countResult[0].total
    const remainingUsers = license.nouser - totalEmployees

    // 3. Block if limit reached
    if (remainingUsers <= 0) {
      return res.status(403).json({
        success: false,
        message: "User limit reached. Please upgrade your license.",
      })
    }

    /* ================= DUPLICATE CHECK ================= */
    const [existingEmployee] = await adminDbPool.execute(
      "SELECT id FROM userlist WHERE email = ? AND loginid = ?",
      [email, adminId]
    )

    if (existingEmployee.length > 0) {
      return res.status(400).json({
        success: false,
        message: "Employee with this email already exists",
      })
    }

    /* ================= INSERT EMPLOYEE ================= */
    const [result] = await adminDbPool.execute(
      "INSERT INTO userlist (loginid, email, sendemail, datecreated) VALUES (?, ?, ?, NOW())",
      [adminId, email, 0]
    )

    const employeeId = result.insertId

    /* ================= UPDATE REMAINING USERS ================= */
    await empCloudDbPool.execute(
      "UPDATE license SET remaininguser = ? WHERE loginid = ?",
      [remainingUsers - 1, adminId]
    )

    /* ================= USER INIT API ================= */
    console.log(`🔄 Calling UserInit API for employee ID: ${employeeId}`)
    const apiResult = await this.callUserInitApi(adminId, employeeId)

    if (!apiResult.success) {
      console.error("❌ UserInit API failed:", apiResult.error)
    }

    /* ================= SEND EMAIL (OPTIONAL) ================= */
    let emailResult = null

    if (sendEmailFlag && nodemailer) {
      emailResult = await this.sendEmployeeEmail(
        adminId,
        employeeId,
        empCloudDbPool
      )

      if (emailResult.success) {
        await adminDbPool.execute(
          "UPDATE userlist SET sendemail = 1, lastupdated = NOW() WHERE id = ?",
          [employeeId]
        )
      }
    }

    /* ================= SUCCESS RESPONSE ================= */
    res.status(201).json({
      success: true,
      message: "Employee added successfully",
      data: {
        id: employeeId,
        email,
        remainingUsers: remainingUsers - 1,
        totalAllowedUsers: license.nouser,
        emailSent: emailResult?.success || false,
      },
    })
  } catch (error) {
    console.error("❌ Add employee error:", error)

    res.status(500).json({
      success: false,
      message: "Failed to add employee",
      error: process.env.NODE_ENV === "development" ? error.message : undefined,
    })
  }
}


  // Upload CSV and process employees
 async uploadCSV(req, res) {
  const adminId = req.user.userId

  try {
    if (!req.file) {
      return res.status(400).json({
        success: false,
        message: "No CSV file uploaded",
      })
    }

    const sendEmailFlag = req.body.sendEmailFlag === "true"
    const databaseConfig = require("../config/database")
    const empCloudDbPool = req.app.locals.empCloudDbPool
    const adminDbPool = databaseConfig.getAdminDbPool(adminId)

    /* ================= LICENSE CHECK ================= */
    const [[license]] = await empCloudDbPool.execute(
      "SELECT nouser FROM license WHERE loginid = ?",
      [adminId]
    )

    if (!license) {
      return res.status(400).json({
        success: false,
        message: "License not found",
      })
    }

    const [countResult] = await adminDbPool.execute(
      "SELECT COUNT(*) AS total FROM userlist WHERE loginid = ?",
      [adminId]
    )

    let remainingUsers = license.nouser - countResult[0].total

    if (remainingUsers <= 0) {
      return res.status(403).json({
        success: false,
        message: "User limit reached. Cannot upload employees.",
      })
    }

    /* ================= CSV PROCESS ================= */
    const results = []
    const errors = []
    const csvData = []

    fs.createReadStream(req.file.path)
      .pipe(csv())
      .on("data", (data) => csvData.push(data))
      .on("end", async () => {
        for (let i = 0; i < csvData.length; i++) {
          if (remainingUsers <= 0) {
            errors.push("License limit reached. Remaining employees skipped.")
            break
          }

          const email = csvData[i].email
          if (!email) {
            errors.push(`Row ${i + 1}: Email missing`)
            continue
          }

          const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
          if (!emailRegex.test(email)) {
            errors.push(`Row ${i + 1}: Invalid email`)
            continue
          }

          const [existing] = await adminDbPool.execute(
            "SELECT id FROM userlist WHERE email = ? AND loginid = ?",
            [email, adminId]
          )

         if (existing.length > 0) {
  errors.push(`Email already exists: ${email}`)
  continue
}


          const [result] = await adminDbPool.execute(
            "INSERT INTO userlist (loginid, email, sendemail, datecreated) VALUES (?, ?, 0, NOW())",
            [adminId, email]
          )

          const employeeId = result.insertId

          // Call UserInit API
          await this.callUserInitApi(adminId, employeeId)

          // Optional Email
          if (sendEmailFlag && nodemailer) {
            const emailResult = await this.sendEmployeeEmail(
              adminId,
              employeeId,
              empCloudDbPool
            )

            if (emailResult.success) {
              await adminDbPool.execute(
                "UPDATE userlist SET sendemail = 1 WHERE id = ?",
                [employeeId]
              )
            }
          }

          remainingUsers--

         results.push({
  email,
  status: "added",
  emailSent: sendEmailFlag === true,
})

        }

        /* ================= UPDATE LICENSE ================= */
        await empCloudDbPool.execute(
          "UPDATE license SET remaininguser = ? WHERE loginid = ?",
          [remainingUsers, adminId]
        )

        fs.unlinkSync(req.file.path)

       res.json({
  success: true,
  message: "CSV processed successfully",
  data: {
    added: results,                    // ✅ ARRAY
    totalAdded: results.length,        // ✅ NUMBER
    skipped: errors.length,            // ✅ NUMBER
    errors,                            // ✅ ARRAY
    remainingUsers,
    totalAllowedUsers: license.nouser,
  },
})

      })
  } catch (error) {
    console.error("❌ CSV upload error:", error)
    res.status(500).json({
      success: false,
      message: "Failed to upload CSV",
    })
  }
}


  // Send email to employee
  async sendEmail(req, res) {
    const { employeeId } = req.params
    const adminId = req.user.userId

    try {
      if (!nodemailer) {
        return res.status(500).json({
          success: false,
          message: "Email functionality not available - nodemailer not installed",
        })
      }

      const databaseConfig = require("../config/database")
      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      // Get employee details from admin's database
      const [employees] = await adminDbPool.execute("SELECT * FROM userlist WHERE id = ? AND loginid = ?", [
        employeeId,
        adminId,
      ])

      if (employees.length === 0) {
        return res.status(404).json({
          success: false,
          message: "Employee not found",
        })
      }

      const emailResult = await this.sendEmployeeEmail(adminId, employeeId, req.app.locals.empCloudDbPool)

      if (emailResult.success) {
        // Update sendemail flag to 1 (sent) in admin's database
        await adminDbPool.execute("UPDATE userlist SET sendemail = 1, lastupdated = NOW() WHERE id = ?", [employeeId])

        return res.json({
          success: true,
          message: "Email sent successfully",
          data: emailResult,
        })
      } else {
        return res.status(500).json({
          success: false,
          message: "Failed to send email",
          error: emailResult.error,
        })
      }
    } catch (error) {
      console.error("Send email error:", error)
      return res.status(500).json({
        success: false,
        message: "Failed to send email",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Resend email to employee
  async resendEmail(req, res) {
    const { employeeId } = req.params
    const adminId = req.user.userId

    try {
      if (!nodemailer) {
        return res.status(500).json({
          success: false,
          message: "Email functionality not available - nodemailer not installed",
        })
      }

      const databaseConfig = require("../config/database")
      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      // Get employee details from admin's database
      const [employees] = await adminDbPool.execute("SELECT * FROM userlist WHERE id = ? AND loginid = ?", [
        employeeId,
        adminId,
      ])

      if (employees.length === 0) {
        return res.status(404).json({
          success: false,
          message: "Employee not found",
        })
      }

      const emailResult = await this.sendEmployeeEmail(adminId, employeeId, req.app.locals.empCloudDbPool)

      if (emailResult.success) {
        // Update sendemail flag to 2 (resent) in admin's database
        await adminDbPool.execute("UPDATE userlist SET sendemail = 2, lastupdated = NOW() WHERE id = ?", [employeeId])

        res.json({
          success: true,
          message: "Email resent successfully",
          data: emailResult,
        })
      } else {
        res.status(500).json({
          success: false,
          message: "Failed to resend email",
          error: emailResult.error,
        })
      }
    } catch (error) {
      console.error("❌ Resend email error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to resend email",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Bulk email functionality
  async bulkEmail(req, res) {
    const { employeeIds } = req.body
    const adminId = req.user.userId

    try {
      if (!nodemailer) {
        return res.status(500).json({
          success: false,
          message: "Email functionality not available - nodemailer not installed",
        })
      }

      if (!employeeIds || !Array.isArray(employeeIds) || employeeIds.length === 0) {
        return res.status(400).json({
          success: false,
          message: "Employee IDs are required",
        })
      }

      const databaseConfig = require("../config/database")
      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      // Get employee details from admin's database
      const placeholders = employeeIds.map(() => "?").join(",")
      const [employees] = await adminDbPool.execute(
        `SELECT * FROM userlist WHERE id IN (${placeholders}) AND loginid = ?`,
        [...employeeIds, adminId],
      )

      if (employees.length === 0) {
        return res.status(404).json({
          success: false,
          message: "No employees found",
        })
      }

      const results = {
        totalProcessed: employees.length,
        successCount: 0,
        failureCount: 0,
        successes: [],
        failures: [],
      }

      // Process each employee
      for (const employee of employees) {
        try {
          const emailResult = await this.sendEmployeeEmail(adminId, employee.id, req.app.locals.empCloudDbPool)

          if (emailResult.success) {
            // Determine the new sendemail status
            let newStatus = 1 // First send
            let action = "Email sent"

            if (employee.sendemail === 1) {
              newStatus = 2 // Resend
              action = "Email resent"
            } else if (employee.sendemail === 2) {
              newStatus = 2 // Keep as resent
              action = "Email resent again"
            }

            // Update sendemail flag in admin's database
            await adminDbPool.execute("UPDATE userlist SET sendemail = ?, lastupdated = NOW() WHERE id = ?", [
              newStatus,
              employee.id,
            ])

            results.successCount++
            results.successes.push({
              id: employee.id,
              email: employee.email,
              action: action,
            })

            console.log(`✅ Bulk email sent to: ${employee.email}`)
          } else {
            results.failureCount++
            results.failures.push({
              id: employee.id,
              email: employee.email,
              error: emailResult.error,
            })

            console.error(`❌ Bulk email failed for: ${employee.email} - ${emailResult.error}`)
          }
        } catch (error) {
          results.failureCount++
          results.failures.push({
            id: employee.id,
            email: employee.email,
            error: error.message,
          })

          console.error(`❌ Bulk email error for: ${employee.email}`, error)
        }
      }

      res.json({
        success: true,
        message: `Bulk email completed. ${results.successCount} sent, ${results.failureCount} failed.`,
        data: results,
      })
    } catch (error) {
      console.error("❌ Bulk email error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to send bulk emails",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Helper method to send employee email
  async sendEmployeeEmail(adminId, employeeId, empCloudDbPool) {
    try {
      if (!nodemailer) {
        throw new Error("Nodemailer not available")
      }

      // Get admin details and organization info
      const [adminData] = await empCloudDbPool.execute(
        "SELECT firstname, lastname, orgid, email FROM useraccounts WHERE id = ?",
        [adminId],
      )

      if (adminData.length === 0) {
        throw new Error("Admin not found")
      }

      const admin = adminData[0]
      const orgId = admin.orgid

      // Get employee details and unique link from admin's database
      const databaseConfig = require("../config/database")
      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      const [employeeData] = await adminDbPool.execute(
        "SELECT email, link FROM userlist WHERE id = ? AND loginid = ?",
        [employeeId, adminId],
      )

      if (employeeData.length === 0) {
        throw new Error("Employee not found in admin database")
      }

      const employee = employeeData[0]
      const employeeEmail = employee.email

      // Use the unique link from the database, or fallback to generic link
      let downloadLink = employee.link
      if (!downloadLink) {
        // Fallback to generic link if unique link is not available
        downloadLink = `/home/empmon/org/${orgId}/EMAgentSetup.zip`
        console.warn(`⚠️ No unique link found for employee ID: ${employeeId}, using fallback link`)
      } else {
        console.log(`✅ Using unique download link for employee ID: ${employeeId}: ${downloadLink}`)
      }

      // Email content
      const emailContent = `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; border: 1px solid #ddd; border-radius: 10px;">
        <div style="text-align: center; margin-bottom: 30px;">
          <h2 style="color: #333; margin-bottom: 10px;">WorkDesQ</h2>
          <p style="color: #666; font-size: 16px;">Welcome to our organization!</p>
        </div>
        
        <div style="background-color: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
          <h3 style="color: #333; margin-top: 0;">Software Installation Required</h3>
          <p style="color: #555; line-height: 1.6;">
            Hello,<br><br>
            You have been added to our WorkDesQ by <strong>${admin.firstname} ${admin.lastname}</strong>.
            To get started, please download and install the monitoring software using your personalized link below.
          </p>
        </div>
        
        <div style="text-align: center; margin: 30px 0;">
          <a href="${downloadLink}" 
             style="background-color: #007bff; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; font-weight: bold; display: inline-block;">
            Download Your Software
          </a>
        </div>
        
        <div style="background-color: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 5px; margin: 20px 0;">
          <h4 style="color: #856404; margin-top: 0;">Installation Instructions:</h4>
          <ol style="color: #856404; line-height: 1.6;">
            <li>Click your personalized download link above</li>
<li>Download and unzip the file</li>
<li>Open the extracted folder and run the WorkDesQSetup Agent file</li>
<li>Follow the installation wizard</li>
<li>The software will start automatically after installation</li>

          </ol>
        </div>
        
     
      </div>
    `

      // Configure email transporter
      const transporter = nodemailer.createTransport({
        host: process.env.EMAIL_HOST,
        port: process.env.EMAIL_PORT,
        secure: false, // true for 465, false for other ports
        auth: {
          user: process.env.EMAIL_USER,
          pass: process.env.EMAIL_PASS,
        },
      })
     

      // Verify transporter configuration
      await transporter.verify()

      // Send email
      const mailOptions = {
        from: `"${admin.firstname} ${admin.lastname}" <${process.env.EMAIL_USER}>`,
        to: employeeEmail,
        subject: "WorkDesQ User Monitoring Software - Installation Required",
        html: emailContent,
      }

      const info = await transporter.sendMail(mailOptions)
      console.log("📨 SMTP accepted:", info.accepted)
console.log("📨 SMTP rejected:", info.rejected)
console.log("📨 Message ID:", info.messageId)

      console.log(`✅ Email sent successfully to ${employeeEmail}:`, info.messageId)

      return {
        success: true,
        messageId: info.messageId,
        downloadLink: downloadLink,
        orgId: orgId,
        employeeId: employeeId,
      }
    } catch (error) {
      console.error("❌ Email sending error:", error)
      return {
        success: false,
        error: error.message,
      }
    }
  }

  // Update employee
  async updateEmployee(req, res) {
    const { employeeId } = req.params
    const { email } = req.body
    const adminId = req.user.userId

    try {
      if (!email) {
        return res.status(400).json({
          success: false,
          message: "Email is required",
        })
      }

      const databaseConfig = require("../config/database")
      const adminDbPool = databaseConfig.getAdminDbPool(adminId)

      // Check if employee exists and belongs to this admin in admin's database
      const [employees] = await adminDbPool.execute("SELECT * FROM userlist WHERE id = ? AND loginid = ?", [
        employeeId,
        adminId,
      ])

      if (employees.length === 0) {
        return res.status(404).json({
          success: false,
          message: "Employee not found",
        })
      }
// 🔒 BLOCK EDIT IF SOFTWARE INSTALLED
const [userDetails] = await adminDbPool.execute(
  "SELECT id FROM userdetails WHERE userid = ? AND loginid = ?",
  [employeeId, adminId]
)

if (userDetails.length > 0) {
  return res.status(403).json({
    success: false,
    message: "Cannot edit employee after software installation",
  })
}
      // Update employee in admin's database
      await adminDbPool.execute("UPDATE userlist SET email = ?, lastupdated = NOW() WHERE id = ? AND loginid = ?", [
        email,
        employeeId,
        adminId,
      ])
await this.callUserInitApi(adminId, employeeId)

      res.json({
        success: true,
        message: "Employee updated successfully",
      })
    } catch (error) {
      console.error("❌ Update employee error:", error)
      res.status(500).json({
        success: false,
        message: "Failed to update employee",
        error: process.env.NODE_ENV === "development" ? error.message : undefined,
      })
    }
  }

  // Delete employee
 async deleteEmployee(req, res) {
  const { employeeId } = req.params
  const adminId = req.user.userId

  try {
    const databaseConfig = require("../config/database")
    const empCloudDbPool = req.app.locals.empCloudDbPool
    const adminDbPool = databaseConfig.getAdminDbPool(adminId)

    // Check employee exists
    const [employee] = await adminDbPool.execute(
      "SELECT id FROM userlist WHERE id = ? AND loginid = ?",
      [employeeId, adminId]
    )

    if (employee.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Employee not found",
      })
    }
// 🔒 BLOCK DELETE IF SOFTWARE INSTALLED
const [userDetails] = await adminDbPool.execute(
  "SELECT id FROM userdetails WHERE userid = ? AND loginid = ?",
  [employeeId, adminId]
)

if (userDetails.length > 0) {
  return res.status(403).json({
    success: false,
    message: "Cannot delete employee after software installation",
  })
}

    // Delete employee
    await adminDbPool.execute(
      "DELETE FROM userlist WHERE id = ?",
      [employeeId]
    )

    // Increase remaining users
    await empCloudDbPool.execute(
      "UPDATE license SET remaininguser = remaininguser + 1 WHERE loginid = ?",
      [adminId]
    )

    res.json({
      success: true,
      message: "Employee deleted successfully",
    })
  } catch (error) {
    console.error("❌ Delete employee error:", error)
    res.status(500).json({
      success: false,
      message: "Failed to delete employee",
    })
  }
}


  // Helper method to call .NET API for user initialization and get unique link
  async callUserInitApi(adminId, employeeId) {
    try {
      if (!fetch) {
        throw new Error("node-fetch not available")
      }

      // const apiUrl = `https://api.workdesq.ai/api/Home/UserInit?param1=${adminId}&param2=${employeeId}`
      const apiUrl = `http://103.148.165.134:5006/api/Home/UserInit?param1=${adminId}&param2=${employeeId}`

      console.log(`🔄 Calling .NET UserInit API: ${apiUrl}`)

      const response = await fetch(apiUrl, {
        method: "GET",
        headers: {
          "Content-Type": "application/json",
          "User-Agent": "Employee-Monitoring-System/1.0",
        },
        timeout: 15000, // 15 second timeout
      })

      const responseText = await response.text()
      console.log(`📊 .NET UserInit API Response Status: ${response.status}`)
      console.log(`📄 .NET UserInit API Response: ${responseText}`)

      if (response.ok) {
        return {
          success: true,
          status: response.status,
          data: responseText,
          url: apiUrl,
        }
      } else {
        return {
          success: false,
          status: response.status,
          error: `API returned status ${response.status}: ${responseText}`,
          url: apiUrl,
        }
      }
    } catch (error) {
      console.error(`❌ .NET UserInit API call error:`, error.message)
      return {
        success: false,
        error: error.message,
        // url: `https://api.workdesq.ai/api/Home/UserInit?param1=${adminId}&param2=${employeeId}`,
         url: `http://103.148.165.134:5006/api/Home/UserInit?param1=${adminId}&param2=${employeeId}`,
      }
    }
  }
}

module.exports = new EmployeeController()
