const { queryMainDb, queryOrgDb, getOrgDbName } = require('../config/database');

const getAdminInfo = async (req, res) => {
  try {
    const { loginid } = req.params;
    
    // Get admin info from useraccounts
    const adminQuery = 'SELECT * FROM useraccounts WHERE id = ?';
    const adminResults = await queryMainDb(adminQuery, [loginid]);
    
    if (adminResults.length === 0) {
      return res.status(404).json({ 
        success: false,
        message: 'Admin not found' 
      });
    }
    
    const admin = adminResults[0];
    
    // Get services info
    const servicesQuery = 'SELECT * FROM product WHERE loginid = ?';
    const servicesResults = await queryMainDb(servicesQuery, [loginid]);
    
    // Calculate active services count
    let activeServices = 0;
    if (servicesResults.length > 0) {
      const service = servicesResults[0];
      for (let i = 1; i <= 8; i++) {
        if (service[`flag${i}`] === 1) {
          activeServices++;
        }
      }
    }
    
    // Get organization database name and employee count
    const dbName = await getOrgDbName(loginid);
    let totalUsers = 0;
    let activeUsers = 0;
    
    try {
      const employeeQuery = 'SELECT COUNT(*) as total, SUM(active) as active FROM userdetails WHERE loginid = ?';
      const employeeResults = await queryOrgDb(dbName, employeeQuery, [loginid]);
      
      if (employeeResults.length > 0) {
        totalUsers = employeeResults[0].total || 0;
        activeUsers = employeeResults[0].active || 0;
      }
    } catch (error) {
      console.warn(`Could not fetch employee data from ${dbName}:`, error.message);
    }
    
    const dashboardData = {
      success: true,
      data: {
        admin: {
          id: admin.id,
          firstname: admin.firstname,
          lastname: admin.lastname,
          email: admin.email,
          displayname: admin.displayname,
          orgid: admin.orgid,
          actype: admin.actype
        },
        stats: {
          totalUsers,
          activeUsers,
          totalServices: 8,
          activeServices
        }
      }
    };
    
    res.json(dashboardData);
  } catch (error) {
    console.error('Error fetching admin info:', error);
    res.status(500).json({ 
      success: false,
      message: 'Internal server error', 
      error: process.env.NODE_ENV === 'development' ? error.message : 'Something went wrong'
    });
  }
};

const getDashboardStats = async (req, res) => {
  try {
    const { loginid } = req.params;
    
    // Get organization database name
    const dbName = await getOrgDbName(loginid);
    
    // Get employee statistics
    const employeeQuery = `
      SELECT 
        COUNT(*) as totalUsers,
        SUM(CASE WHEN active = 1 THEN 1 ELSE 0 END) as activeUsers,
        SUM(CASE WHEN connected = 1 THEN 1 ELSE 0 END) as connectedUsers
      FROM userdetails 
      WHERE loginid = ?
    `;
    
    const employeeResults = await queryOrgDb(dbName, employeeQuery, [loginid]);
    
    // Get service statistics
    const serviceQuery = 'SELECT * FROM product WHERE loginid = ?';
    const serviceResults = await queryMainDb(serviceQuery, [loginid]);
    
    let activeServices = 0;
    if (serviceResults.length > 0) {
      const service = serviceResults[0];
      for (let i = 1; i <= 8; i++) {
        if (service[`flag${i}`] === 1) {
          activeServices++;
        }
      }
    }
    
    const stats = {
      totalUsers: employeeResults[0]?.totalUsers || 0,
      activeUsers: employeeResults[0]?.activeUsers || 0,
      connectedUsers: employeeResults[0]?.connectedUsers || 0,
      totalServices: 8,
      activeServices
    };
    
    res.json({
      success: true,
      data: stats
    });
    
  } catch (error) {
    console.error('Error fetching dashboard stats:', error);
    res.status(500).json({ 
      success: false,
      message: 'Internal server error', 
      error: process.env.NODE_ENV === 'development' ? error.message : 'Something went wrong'
    });
  }
};

module.exports = {
  getAdminInfo,
  getDashboardStats
};