const express = require("express")
const router = express.Router()
const path = require("path")
const fs = require("fs")
const employeeController = require("../controllers/employeeController")
const authMiddleware = require("../middleware/authMiddleware.js")

// Try to require multer, handle gracefully if not available
let multer, upload
try {
  multer = require("multer")

  // Ensure uploads directory exists
  const uploadsDir = path.join(__dirname, "..", "uploads", "csv")
  if (!fs.existsSync(uploadsDir)) {
    fs.mkdirSync(uploadsDir, { recursive: true })
    console.log("📁 Created uploads/csv directory")
  }

  // Configure multer for CSV file uploads
  const storage = multer.diskStorage({
    destination: (req, file, cb) => {
      cb(null, uploadsDir)
    },
    filename: (req, file, cb) => {
      cb(null, `employees-${Date.now()}-${Math.round(Math.random() * 1e9)}.csv`)
    },
  })

  upload = multer({
    storage: storage,
    limits: {
      fileSize: 5 * 1024 * 1024, // 5MB limit
    },
    fileFilter: (req, file, cb) => {
      if (file.mimetype === "text/csv" || file.originalname.endsWith(".csv")) {
        cb(null, true)
      } else {
        cb(new Error("Only CSV files are allowed"))
      }
    },
  })

  console.log("✅ Multer configured successfully")
} catch (e) {
  console.warn("⚠️ multer not available - CSV upload functionality disabled")
  // Create a dummy middleware that returns an error
  upload = {
    single: () => (req, res, next) => {
      res.status(500).json({
        success: false,
        message: "File upload functionality not available - multer not installed",
      })
    },
  }
}

// All routes require authentication
router.use(authMiddleware.verifyToken)

// Database check route
router.get("/check-database", employeeController.checkAdminDatabase)

// Employee management routes
router.get("/", employeeController.getEmployees)
router.post("/add", employeeController.addEmployee)
router.post("/upload-csv", upload.single("csvFile"), employeeController.uploadCSV)
router.post("/bulk-email", employeeController.bulkEmail)
router.post("/:employeeId/send-email", employeeController.sendEmail)
router.post("/:employeeId/resend-email", employeeController.resendEmail)
router.put("/:employeeId", employeeController.updateEmployee)
router.delete("/:employeeId", employeeController.deleteEmployee)

module.exports = router
