const { queryOrgDb, getOrgDbName } = require('../config/database');

const getUserActivity = async (req, res) => {
  try {
    const { userid } = req.params;
    const { loginid, limit = 100, offset = 0 } = req.query;
    
    if (!loginid) {
      return res.status(400).json({
        success: false,
        message: 'loginid is required as query parameter'
      });
    }
    
    // Get organization database name
    const dbName = await getOrgDbName(loginid);
    
    // Get activity data for the user
    const query = `
      SELECT 
        id, 
        loginid, 
        userid, 
        duration, 
        status, 
        startdatetime, 
        enddatetime, 
        timecreated
      FROM userdata81 
      WHERE userid = ? AND loginid = ?
      ORDER BY startdatetime DESC 
      LIMIT ? OFFSET ?
    `;
    
    const activities = await queryOrgDb(dbName, query, [userid, loginid, parseInt(limit), parseInt(offset)]);
    
    // Get summary statistics
    const summaryQuery = `
      SELECT 
        SUM(CASE WHEN status = 'Working' THEN duration ELSE 0 END) as workingTime,
        SUM(CASE WHEN status = 'Idle' THEN duration ELSE 0 END) as idleTime,
        SUM(duration) as totalTime,
        COUNT(*) as totalSessions,
        COUNT(CASE WHEN status = 'Working' THEN 1 END) as workingSessions,
        COUNT(CASE WHEN status = 'Idle' THEN 1 END) as idleSessions
      FROM userdata81 
      WHERE userid = ? AND loginid = ?
      AND DATE(startdatetime) = CURDATE()
    `;
    
    const summaryResults = await queryOrgDb(dbName, summaryQuery, [userid, loginid]);
    const summary = summaryResults[0] || {};
    
    // Calculate percentages
    const totalTime = parseFloat(summary.totalTime) || 0;
    const workingTime = parseFloat(summary.workingTime) || 0;
    const idleTime = parseFloat(summary.idleTime) || 0;
    
    const workingPercentage = totalTime > 0 ? ((workingTime / totalTime) * 100).toFixed(1) : 0;
    const idlePercentage = totalTime > 0 ? ((idleTime / totalTime) * 100).toFixed(1) : 0;
    
    res.json({
      success: true,
      data: {
        activities: activities.map(activity => ({
          ...activity,
          duration: parseFloat(activity.duration)
        })),
        summary: {
          workingTime: workingTime.toFixed(1),
          idleTime: idleTime.toFixed(1),
          totalTime: totalTime.toFixed(1),
          workingPercentage: parseFloat(workingPercentage),
          idlePercentage: parseFloat(idlePercentage),
          totalSessions: summary.totalSessions || 0,
          workingSessions: summary.workingSessions || 0,
          idleSessions: summary.idleSessions || 0
        }
      }
    });
    
  } catch (error) {
    console.error('Error fetching user activity:', error);
    res.status(500).json({ 
      success: false,
      message: 'Internal server error', 
      error: process.env.NODE_ENV === 'development' ? error.message : 'Something went wrong'
    });
  }
};

const getActivitySummary = async (req, res) => {
  try {
    const { userid } = req.params;
    const { loginid, startDate, endDate, groupBy = 'day' } = req.query;
    
    if (!loginid) {
      return res.status(400).json({
        success: false,
        message: 'loginid is required as query parameter'
      });
    }
    
    const dbName = await getOrgDbName(loginid);
    
    let dateFilter = '';
    const params = [userid, loginid];
    
    if (startDate && endDate) {
      dateFilter = 'AND DATE(startdatetime) BETWEEN ? AND ?';
      params.push(startDate, endDate);
    } else {
      // Default to last 7 days
      dateFilter = 'AND DATE(startdatetime) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)';
    }
    
    let groupByClause = 'DATE(startdatetime)';
    let selectClause = 'DATE(startdatetime) as activity_date';
    
    if (groupBy === 'hour') {
      groupByClause = 'DATE(startdatetime), HOUR(startdatetime)';
      selectClause = 'DATE(startdatetime) as activity_date, HOUR(startdatetime) as activity_hour';
    }
    
    const query = `
      SELECT 
        ${selectClause},
        status,
        SUM(duration) as total_duration,
        COUNT(*) as session_count,
        AVG(duration) as avg_duration
      FROM userdata81 
      WHERE userid = ? AND loginid = ? ${dateFilter}
      GROUP BY ${groupByClause}, status 
      ORDER BY activity_date DESC, ${groupBy === 'hour' ? 'activity_hour DESC,' : ''} status
    `;
    
    const results = await queryOrgDb(dbName, query, params);
    
    res.json({
      success: true,
      data: results.map(row => ({
        ...row,
        total_duration: parseFloat(row.total_duration),
        avg_duration: parseFloat(row.avg_duration)
      }))
    });
    
  } catch (error) {
    console.error('Error fetching activity summary:', error);
    res.status(500).json({ 
      success: false,
      message: 'Internal server error', 
      error: process.env.NODE_ENV === 'development' ? error.message : 'Something went wrong'
    });
  }
};

const getProductivityReport = async (req, res) => {
  try {
    const { loginid } = req.params;
    const { startDate, endDate } = req.query;
    
    const dbName = await getOrgDbName(loginid);
    
    let dateFilter = '';
    const params = [loginid];
    
    if (startDate && endDate) {
      dateFilter = 'AND DATE(ud.startdatetime) BETWEEN ? AND ?';
      params.push(startDate, endDate);
    } else {
      dateFilter = 'AND DATE(ud.startdatetime) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)';
    }
    
    const query = `
      SELECT 
        ue.id as employee_id,
        ue.email,
        ue.devicename,
        SUM(CASE WHEN ud.status = 'Working' THEN ud.duration ELSE 0 END) as working_time,
        SUM(CASE WHEN ud.status = 'Idle' THEN ud.duration ELSE 0 END) as idle_time,
        SUM(ud.duration) as total_time,
        COUNT(ud.id) as total_sessions,
        ROUND(
          (SUM(CASE WHEN ud.status = 'Working' THEN ud.duration ELSE 0 END) / 
           NULLIF(SUM(ud.duration), 0)) * 100, 2
        ) as productivity_percentage
      FROM userdetails ue
      LEFT JOIN userdata81 ud ON ue.id = ud.userid AND ue.loginid = ud.loginid
      WHERE ue.loginid = ? ${dateFilter}
      GROUP BY ue.id, ue.email, ue.devicename
      ORDER BY productivity_percentage DESC
    `;
    
    const results = await queryOrgDb(dbName, query, params);
    
    res.json({
      success: true,
      data: results.map(row => ({
        ...row,
        working_time: parseFloat(row.working_time) || 0,
        idle_time: parseFloat(row.idle_time) || 0,
        total_time: parseFloat(row.total_time) || 0,
        productivity_percentage: parseFloat(row.productivity_percentage) || 0
      }))
    });
    
  } catch (error) {
    console.error('Error fetching productivity report:', error);
    res.status(500).json({ 
      success: false,
      message: 'Internal server error', 
      error: process.env.NODE_ENV === 'development' ? error.message : 'Something went wrong'
    });
  }
};

module.exports = {
  getUserActivity,
  getActivitySummary,
  getProductivityReport
};